// Create a tunnel between two lookup services

package corejini.chapter9;

import net.jini.core.lookup.ServiceEvent;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceItem;
import net.jini.core.lookup.ServiceMatches;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.EventRegistration;
import net.jini.core.discovery.LookupLocator;
import net.jini.core.entry.Entry;
import net.jini.core.lease.Lease;
import net.jini.lookup.entry.Name;
import net.jini.lookup.entry.ServiceInfo;
import net.jini.lookup.JoinManager;
import net.jini.lookup.ServiceIDListener;
import net.jini.lease.LeaseRenewalManager;
import net.jini.discovery.LookupDiscoveryManager;
import net.jini.discovery.DiscoveryGroupManagement;
import java.io.Serializable;
import java.io.IOException;
import java.util.HashMap;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;

class TunnelServiceProxy implements Serializable {
    // Currently this is an "empty" proxy.
}

public class TunnelService {
    protected JoinManager join;
    protected ServiceTemplate tmpl = 
	new ServiceTemplate(null, null, null);
    protected LeaseRenewalManager leaseMgr = 
	new LeaseRenewalManager();
    protected HashMap leases = new HashMap();
    protected HashMap joinManagers = new HashMap();
    protected LookupLocator toLoc;
    protected ServiceRegistrar fromReg;
    protected int transitions = 
	ServiceRegistrar.TRANSITION_MATCH_NOMATCH |
        ServiceRegistrar.TRANSITION_NOMATCH_MATCH |
        ServiceRegistrar.TRANSITION_MATCH_MATCH;
    protected Listener listener;

    // Listen for changes in the source lookup service
    class Listener extends UnicastRemoteObject 
		implements RemoteEventListener { 
        public Listener() throws RemoteException {
	}

        public void notify(RemoteEvent ev) throws RemoteException {
            if (!(ev instanceof ServiceEvent)) {
                return;
            }
        
            ServiceEvent sev = (ServiceEvent) ev;
            ServiceItem[] items = new ServiceItem[1];
                items[0] = sev.getServiceItem();
        
            switch (sev.getTransition()) {
            case ServiceRegistrar.TRANSITION_MATCH_NOMATCH:
                withdrawServices(items);
                break;
            case ServiceRegistrar.TRANSITION_NOMATCH_MATCH:
            case ServiceRegistrar.TRANSITION_MATCH_MATCH:
                // new service has shown up, or an old one 
		// has changed--push it to the dest.  if 
		// it's a change we'll overwrite
                pushServices(items);
                break;
            }
        }
    }
    
    public TunnelService() throws RemoteException {
	listener = new Listener();

	if (System.getSecurityManager() == null) {
	    System.setSecurityManager(
		    new RMISecurityManager());
	}
    }
    
    public void createTunnel(String from, 
			     String to) throws Exception {
        LookupLocator fromLoc = new LookupLocator(from);
        toLoc = new LookupLocator(to);
        fromReg = fromLoc.getRegistrar();
        
        publishProxies(fromLoc, toLoc);
        solicitEvents();
        ServiceItem[] fromServices = getServices();
        pushServices(fromServices);
    }
    
    protected ServiceItem[] getServices() throws RemoteException {
        ServiceMatches matches = fromReg.lookup(tmpl, 
						Integer.MAX_VALUE);
        return matches.items;
    }

    protected void publishProxies(LookupLocator from, 
				  LookupLocator to) 
        throws IOException {
        Name name = new Name("Tunnel service");
        ServiceInfo info = new ServiceInfo("Tunnel service",
                                           "Prentice-Hall",
                                           "Your local bookseller",
                                           "v1.0",
                                           null, null);
        Entry[] entries = new Entry[2];
        entries[0] = name;
        entries[1] = info;
        
        LookupLocator[] locators = new LookupLocator[2];
        locators[0] = from;
        locators[1] = to;
        
        String[] groups = DiscoveryGroupManagement.NO_GROUPS;
        
        LookupDiscoveryManager luMgr =
            new LookupDiscoveryManager(groups, locators, null);
        join = new JoinManager(new TunnelServiceProxy(), 
			       entries, (ServiceIDListener) null, 
                               luMgr, leaseMgr);
        
        System.out.println("Join manager started...");
    }
    
    protected void solicitEvents() throws IOException,
            ClassNotFoundException {
        EventRegistration er = fromReg.notify(tmpl, 
					      transitions, 
					      listener,
                                              null, 
					      10 * 60 * 1000);
        
        leases.put(fromReg.getServiceID(), er.getLease());
        leaseMgr.renewUntil(er.getLease(), Lease.ANY, null);
    }
    
    protected void pushServices(ServiceItem[] items) {
        for (int i=0 ; i<items.length ; i++) {
            try {
                LookupLocator[] toArray = 
			new LookupLocator[] { toLoc };
                System.out.println("Pushing service to " + 
				   toLoc + ": " +
                                   items[i].service);
                LookupDiscoveryManager luMgr =
                    new LookupDiscoveryManager(
                            DiscoveryGroupManagement.NO_GROUPS,
                            toArray, null);
                JoinManager j = 
		    new JoinManager(items[i].service,
                                    items[i].attributeSets,
                                    items[i].serviceID,
                                    luMgr,
                                    leaseMgr);
                joinManagers.put(items[i].serviceID, j);
            } catch (Exception ex) {
                System.err.println("Couldn't tunnel service: " +
                                   ex.getMessage());
            }
        }
    }
    
    protected void withdrawServices(ServiceItem[] items) {
        for (int i=0 ; i<items.length ; i++) {
            try {
                System.out.println("Withdrawing service " + 
				   items[i].service);
                JoinManager j = (JoinManager) 
                    joinManagers.get(items[i].serviceID);
                j.terminate();
                joinManagers.remove(items[i].serviceID);
            } catch (Exception ex) {
                System.err.println("Couldn't withdraw service: " +
                                   ex.getMessage());
            }
        }
    }
    
    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: TunnelService " +
			       "<from_url> <to_url>");
            System.exit(1);
        }
        
        String from = args[0], to = args[1];
        
        try {
            TunnelService tunnel = new TunnelService();
            tunnel.createTunnel(from, to);
        } catch (Exception ex) {
            System.err.println("Error creating tunnel: " + 
			       ex.getMessage());
            System.exit(1);
        }
    }
}
