// This is a simple DiscoveryListener that will
// print when it receives discovery events.

package corejini.chapter6;

import net.jini.discovery.DiscoveryManagement;
import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.DiscoveryEvent;
import net.jini.discovery.DiscoveryGroupManagement;
import net.jini.core.lookup.ServiceRegistrar;
import java.rmi.RemoteException;

public class PrintingListener 
    implements DiscoveryListener {
    protected DiscoveryManagement dm;
    
    // The PrintingListener keeps a handle to a
    // DiscoveryManagement instance.  It will
    // use this to discard any lookup services that
    // start to raise exceptions.  This allows them to
    // be discovered again.
    public PrintingListener(DiscoveryManagement dm) {
        this.dm = dm;
    }
    
    // Print out some simple debugging information
    // when a lookup service is found.
    public void discovered(DiscoveryEvent ev) {
        ServiceRegistrar[] regs = ev.getRegistrars();
        
        for (int i=0 ; i<regs.length ; i++) {
            try {
                System.out.println("Discovered: ");
                System.out.println("\tURL:    " + 
                                   getURL(regs[i]));
                System.out.println("\tID:     " +  
                                   regs[i].getServiceID());
                System.out.println("\tGroups: " + 
                                   getGroups(regs[i]));
            } catch (RemoteException ex) {
                // If the lookup service raises a remote
                // exception, discard it so that it can
                // be re-discovered later.
                System.err.println("Error: " + 
                                   ex.getMessage());
                dm.discard(regs[i]);
            }
        }
    }
            
    // Print out details of discarded lookup services.
    public void discarded(DiscoveryEvent ev) {
        ServiceRegistrar[] regs = ev.getRegistrars();
        
        for (int i=0 ; i<regs.length ; i++) {
             System.out.println("Discarded: " +
			        regs[i]);
        }
    }
    
    // A helper method to get the URL for a 
    // lookup service.
    static String getURL(ServiceRegistrar reg) 
        throws RemoteException {
        return reg.getLocator().toString();
    }
    
    // A helper method to get the groups that a
    // lookup service is a member of.
    static String getGroups(ServiceRegistrar reg)
        throws RemoteException {
        String[] groups = reg.getGroups();

	if (groups == DiscoveryGroupManagement.ALL_GROUPS) {
	    return "<all groups>";
	}
        
        if (groups == DiscoveryGroupManagement.NO_GROUPS) {
            return "<none>";
        }
        
        StringBuffer buf = new StringBuffer();
        for (int i=0 ; i<groups.length ; i++) {
            if (groups[i] == null) {
		// This should never happen.  The Discovery
		// libraries should exclude all nulls.
                buf.append("NULL ");
            } else if (groups[i].equals("")) {
                buf.append("PUBLIC ");
            } else {
                buf.append(groups[i]);
            }
        }
        
        return buf.toString();
    }
}
