// Extend HelloWorldService to renew its service
// registration leases.

package corejini.chapter5;

import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceRegistration;
import net.jini.core.lease.Lease;
import net.jini.core.lease.UnknownLeaseException;
import java.util.ArrayList;
import java.util.Iterator;
import java.io.IOException;
import java.rmi.RemoteException;

public class HelloWorldServiceWithLeases 
    extends HelloWorldService {
    protected Thread leaseThread = null;
    
    public HelloWorldServiceWithLeases() 
        throws IOException {
    }
    
    // Not only register, but also cause the lease
    // thread to wake up.
    protected void registerWithLookup(ServiceRegistrar 
                                      lusvc) {
        super.registerWithLookup(lusvc);
        leaseThread.interrupt();
    }
        
    // run now maintains our leases
    public void run() {
        while (true) {
            try {
                long sleepTime = computeSleepTime();
                Thread.sleep(sleepTime);
                renewLeases();
            } catch (InterruptedException ex) {
            }
        }
    }
    
    // Figure out how long to sleep.
    protected synchronized long computeSleepTime() {
        long soonestExpiration = Long.MAX_VALUE;
        Iterator iter = registrations.keySet().iterator();
        while (iter.hasNext()) {
            ServiceRegistrar lu = 
                (ServiceRegistrar) iter.next();
            Lease l = ((ServiceRegistration) 
                registrations.get(lu)).getLease();
            if (l.getExpiration() - (20 * 1000) < soonestExpiration) {
                soonestExpiration = l.getExpiration() - (20 * 1000);
            }
        }

        long now = System.currentTimeMillis();
        
        if (now >= soonestExpiration) {
            return 0;
        } else {
            return soonestExpiration - now;
        }
    }

    // Do the work of lease renewal.
    protected synchronized void renewLeases() {
        long now = System.currentTimeMillis();
        ArrayList deadLeases = new ArrayList();
    
        Iterator keys = registrations.keySet().iterator();
        while (keys.hasNext()) {
            ServiceRegistrar lu = 
                (ServiceRegistrar) keys.next();
            ServiceRegistration r = 
                (ServiceRegistration) registrations.get(lu);
            Lease l = r.getLease();
            if (now <= l.getExpiration() &&
                now >= l.getExpiration() - (20 * 1000)) {
                try {
                    System.out.println("Renewing lease.");
                    l.renew(LEASE_TIME);
                } catch (Exception ex) {
                    System.err.println("Couldn't renew lease: " +
                                       ex.getMessage());
                    deadLeases.add(lu);
                }
            }
        }
        
        // clean up after any leases that died
        for (int i=0, size=deadLeases.size() ; i<size ; i++) {
            registrations.remove(deadLeases.get(i));
        }
    }
    
    // Create the service and start the leasing
    // thread.
    public static void main(String args[]) {
        try {
            HelloWorldServiceWithLeases hws = 
                new HelloWorldServiceWithLeases();
            hws.leaseThread = new Thread(hws);
            hws.leaseThread.start();
        } catch (IOException ex) {
            System.out.println("Couldn't create service: " +
                               ex.getMessage());
        }
    }
}
