// A version of HelloWorldService that creates a "back end"
// remote object for the proxy to communicate with.

package corejini.chapter5;

import net.jini.core.lookup.ServiceItem;
import java.io.IOException;
import java.io.Serializable;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;

// HelloWorldServiceBackend is the "wrapper" application that
// creates a remote object and publishes a proxy that can refer
// to it.

public class HelloWorldServiceBackend 
    extends HelloWorldServiceWithLeases {
    
    // Backend is the RMI server object that receives remote
    // method invocations from the proxy.  It simply cycles
    // through a set of strings each time it is called.
    static class Backend extends UnicastRemoteObject 
        implements BackendProtocol {
        protected int nextMessage = 0;
        protected String[] messages = { "Hello, World", 
                                        "Goodbye, Cruel World",
                                        "What's Up, Doc?" };
        
        Backend() throws RemoteException {
            super();
        }
        
        public String fetchString() throws RemoteException {
            System.out.println("Back-end: fetching string");
            String str =  messages[nextMessage];
            nextMessage = (nextMessage + 1) % messages.length;
            return str;
        }
    }

    // Constructor doesn't have to do anything
    // except let the superclass constructor run.
    public HelloWorldServiceBackend() throws IOException {
    }

    // Since we're using a different proxy we have to reimplement
    // createProxy().  This new version creates a remote server
    // object that will receive method invocations from the proxy,
    // and creates a HelloWorldServiceProxy2 object that
    // refers to it.
    protected HelloWorldServiceInterface createProxy() {
        try {
            BackendProtocol backend = new Backend();
            return new HelloWorldServiceProxy2(backend);
        } catch (RemoteException ex) {
            System.err.println("Error creating backend: " +
                               ex.getMessage());
            ex.printStackTrace();
            System.exit(1);
            return null;        // NOT REACHED
        }
    }
    
    // Main creates the wrapper and starts the lease
    // thread.
    public static void main(String args[]) {
        try {
            HelloWorldServiceBackend hws = 
                new HelloWorldServiceBackend();
            hws.leaseThread = new Thread(hws);
            hws.leaseThread.start();
        } catch (IOException ex) {
            System.out.println("Couldn't create service: " +
                               ex.getMessage());
        }
    }
}
