//
// A client that uses the three utility services to
// track discoveries.
//

package corejini.chapter19;

import net.jini.lookup.ServiceDiscoveryManager;
import net.jini.event.EventMailbox;
import net.jini.lease.LeaseRenewalService;
import net.jini.lease.LeaseRenewalSet;
import net.jini.discovery.LookupDiscoveryService;
import net.jini.discovery.LookupDiscoveryRegistration;
import net.jini.discovery.DiscoveryGroupManagement;
import net.jini.core.lookup.ServiceItem;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import java.rmi.RMISecurityManager;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.io.IOException;

public class Collector {
    private ServiceDiscoveryManager luMgr;
    private LookupDiscoveryService lds;
    private EventMailbox ems;
    private LeaseRenewalService lrs;
    private LeaseRenewalSet leaseSet;
    
    public Collector() throws IOException {
        luMgr = new ServiceDiscoveryManager(null, null);
    }
    
    // Find all of the three utility services
    public void findUtilityServices() 
        throws Exception {
        
        // service items for the three services
        ServiceItem ldsItem, emsItem, lrsItem;
        ServiceTemplate tmpl;
        Class[] types = new Class[1];
        long waitDur = 1 * 60 * 1000;        // 1 minute
        
        try {
            types[0] = LookupDiscoveryService.class;
            tmpl = new ServiceTemplate(null, types, null);
            ldsItem = luMgr.lookup(tmpl, null, waitDur);
        } catch (InterruptedException ex) {
            System.err.println("No LookupDiscovery service before timeout");
            throw ex;
        }
        
        try {
            types[0] = EventMailbox.class;
            tmpl = new ServiceTemplate(null, types, null);
            emsItem = luMgr.lookup(tmpl, null, waitDur);
        } catch (InterruptedException ex) {
            System.err.println("No EventMailbox service before timeout");
            throw ex;
        }
        
        try {
            types[0] = LeaseRenewalService.class;
            tmpl = new ServiceTemplate(null, types, null);
            lrsItem = luMgr.lookup(tmpl, null, waitDur);
        } catch (InterruptedException ex) {
            System.err.println("No LeaseRenewal service before timeout");
            throw ex;
        }
        
        // At this point, we've got all the service items.
        // Do some checking to make sure there aren't
        // codebase problems.
        
        if (ldsItem.service == null) {
            throw new Exception("No proxy for LookupDiscovery service. " +
                                "Check for codebase problems?");
        } else {
            lds = (LookupDiscoveryService) ldsItem.service;
        }
        
        if (emsItem.service == null) {
            throw new Exception("No proxy for EventMailbox service. " +
                                "Check for codebase problems?");
        } else {
            ems = (EventMailbox) emsItem.service;
        }
        
        if (lrsItem.service == null) {
            throw new Exception("No proxy for LeaseRenewal service. " +
                                "Check for codebase problems?");
        } else {
            lrs = (LeaseRenewalService) lrsItem.service;
        }
    }
    
    // create a lease renewal set that we can use to
    // stuff leases into.  also sign up to receive
    // informational events.
    public void setupLeaseRenewals() 
        throws RemoteException {
        long duration = 2 * 60 * 60 * 1000;     // 2 hours
        leaseSet = lrs.createLeaseRenewalSet(duration);
        
        leaseSet.setExpirationWarningListener(
            new LogEventListener("Lease set lease about to expire"),
            1 * 60 * 1000,    // 1 minute of warning
            null);            // handback object
        
        leaseSet.setRenewalFailureListener(
            new LogEventListener("Couldn't renew lease"),
            null);            // handback object
    }
    
    public void setupEventMailbox() 
        throws RemoteException {
    }
    
    public void setupDiscovery() 
        throws RemoteException {
        long duration = 2 * 60 * 60 * 1000;    // 2 hours
        LookupDiscoveryRegistration luReg =
            lds.register(DiscoveryGroupManagement.ALL_GROUPS, 
                         null,         // no locators
                         null,         // PLUG MAILBOX IN HERE  !!!
                         null,         // handback object
                         duration);
        
        // Add the lease for the registration to the
        // LeaseSet from the LeaseRenewalService so
        // that it will be renewed.
        leaseSet.renewFor(luReg.getLease(), duration);
    }
    
    // Must rmic this class, since it contains a
    // remote method.  This is a simple event listener
    // that simply prints out a message when an
    // event is received.
    class LogEventListener extends UnicastRemoteObject
        implements RemoteEventListener {
        private String msg;
        
        LogEventListener(String msg) throws RemoteException {
            this.msg = msg;
        }
        
        public void notify(RemoteEvent evt) {
            System.out.println(msg + " " + evt);
        }
    }
    
    public static void main(String[] args) {
        if (System.getSecurityManager() == null) {
            System.setSecurityManager(
                    new RMISecurityManager());
        }
        
        try {
            Collector collector = new Collector();
            
            collector.findUtilityServices();
            
            collector.setupLeaseRenewals();
            collector.setupEventMailbox();
            collector.setupDiscovery();
            
        } catch (Exception ex) {
            System.err.println("Error: " + ex.getMessage());
            ex.printStackTrace();
            System.exit(1);
        }
    }
}

