// Safely move a set of Entry objects from one JavaSpace
// to another.

package corejini.chapter18;

import net.jini.space.JavaSpace;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceMatches;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceID;
import net.jini.core.entry.Entry;
import net.jini.core.lease.Lease;
import net.jini.core.transaction.Transaction;
import net.jini.core.transaction.TransactionFactory;
import net.jini.core.transaction.CannotCommitException;
import net.jini.core.transaction.UnknownTransactionException;
import net.jini.core.transaction.server.TransactionManager;
import net.jini.lease.LeaseRenewalManager;
import net.jini.discovery.LookupDiscovery;
import net.jini.discovery.DiscoveryGroupManagement;
import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.DiscoveryEvent;
import net.jini.lookup.entry.Name;
import java.util.ArrayList;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.io.IOException;

public class EntryMover implements Runnable {
    protected Class[] spaceTypes = { JavaSpace.class };
    protected ServiceTemplate spaceTmpl = 
        new ServiceTemplate(null, spaceTypes, null);
    protected Class[] txnTypes = { TransactionManager.class };
    protected ServiceTemplate txnTmpl = 
        new ServiceTemplate(null, txnTypes, null);
    protected ArrayList regs = new ArrayList();
    protected LeaseRenewalManager leaseMgr =
        new LeaseRenewalManager();
    protected JavaSpace js1 = null, js2 = null;
    protected TransactionManager txn = null;
    // This is the ID of any already-found JavaSpace.
    // We need to keep it to weed out duplicates.
    protected ServiceID id = null;

    // An inner class for discovery.
    class Discoverer implements DiscoveryListener {
        public void discovered(DiscoveryEvent ev) {
            System.out.println("Discovered!");
            ServiceRegistrar[] newregs = ev.getRegistrars();
            for (int i=0, size=newregs.length ; i<size ; i++) {
                if (!regs.contains(newregs[i])) {
                    regs.add(newregs[i]);
                    findServices((ServiceRegistrar) newregs[i]);
                }
            }
        }
        public void discarded(DiscoveryEvent ev) {
        }
    }
    
    public EntryMover() throws IOException {
        LookupDiscovery disco = new 
            LookupDiscovery(DiscoveryGroupManagement.ALL_GROUPS);
        disco.addDiscoveryListener(new Discoverer());
    }

    // Find all the services we'll need, and then call
    // doit()!
    protected void findServices(ServiceRegistrar reg) {
        if (js1 != null && js2 != null && txn != null ) {
            return;
        }
        
        ServiceMatches matches = null;
        
        if (js1 == null || js2 == null) {
            System.out.println("Searching for JavaSpace");
            try {
                matches = reg.lookup(spaceTmpl, 
                                     Integer.MAX_VALUE);
            } catch (RemoteException ex) {
                System.err.println("Error doing lookup: " +
                                   ex.getMessage());
                return;
            }
        
            for (int i=0 ; i<matches.totalMatches ; i++) {
                // If we haven't found a space yet, or if
		// this is a  new space, then use it.
                if (js1 == null) {
                    js1 = (JavaSpace) matches.items[i].service;
                    System.out.println("Found first JavaSpace!");
                    id = matches.items[i].serviceID;
                } else if (js2 == null &&
                           !id.equals(matches.items[i].serviceID)) {
                    js2 = (JavaSpace) matches.items[i].service;
                    System.out.println("Found second JavaSpace!");
                }
            }
        }
        
        if (txn == null) {
            System.out.println("Searching for txn mgr");
            try {
                txn = (TransactionManager) reg.lookup(txnTmpl);
                if (txn != null) {
                    System.out.println("Found it!");
                }
            } catch (RemoteException ex) {
                System.err.println("Error: " +
                                   ex.getMessage());
                ex.printStackTrace();
            }
        }
        
        // If we've got two spaces and a transaction
        // manager, then do it!
        if (js1 != null && js2 != null && txn != null)
            doit();
    }
   
    // Load up the first space with some data, and then
    // move it out under transactions.
    protected void doit() {
        System.out.println("Writing some data to first space");
        for (int i=0 ; i<10 ; i++) {
            Name name = new Name(Integer.toString(i));
            try {
                Lease l = js1.write(name, null, 60 * 1000);
                System.out.println("Wrote: " + name);
                leaseMgr.renewUntil(l, Lease.ANY, null);
            } catch (Exception ex) {
                System.err.println("Trouble writing: " +
                                   ex.getMessage());
                ex.printStackTrace();
            }
        }
            
        System.out.println("Moving to second space");
            
        Name nameTmpl = new Name(null);
            
        Entry e = null;
        boolean done = false;
        int idx=0;
        ArrayList uncommits = new ArrayList();
            
        while (!done) {
            try {
                System.out.println("Creating transaction");
            
                Transaction.Created c = 
                    TransactionFactory.create(txn,
                                              10 * 60 * 1000);
                                          
                // Manage the lease on the transaction,
                // just in case.
                leaseMgr.renewUntil(c.lease, Lease.ANY, null);
            
                // Take under a transaction
                e = js1.takeIfExists(nameTmpl, c.transaction, JavaSpace.NO_WAIT);
                
                if (e == null) {
                    done = true;
                    continue;
                }
                
                // Write under the same transaction
                System.out.println("Moving: " + e);
                Lease l = js2.write(e, c.transaction, 60 * 1000);
                leaseMgr.renewUntil(l, Lease.ANY, null);
                
                // Commit!
                if (idx++ % 2 == 0) {
                    System.out.println("Committing!");
                    c.transaction.commit();
                } else {
                    uncommits.add(c.transaction);
                }
            } catch (RemoteException ex) {
                System.err.println("Trouble communicating: " + 
                                   ex.getMessage());
            } catch (CannotCommitException ex) {
                System.err.println("Couldn't commit: " +
                                   ex.getMessage());
            } catch (UnknownTransactionException ex) {
                System.err.println("Unknown transaction: " +
                                   ex.getMessage());
            } catch (Exception ex) {
                System.err.println("Other problem: " +
                                   ex.getMessage());
            }
        }

        // Dump out the contents of the spaces, then
        // commit the remaining transactions, then
        // dump again.

        System.out.println("Contents of space 1: ");
        dumpContents(js1);
            
        System.out.println("Contents of space 2: ");
        dumpContents(js2);

        System.out.println("Committing remaining transactions.");
        for (int i=0, size=uncommits.size() ; i<size ; i++) {
            Transaction t = (Transaction)
                uncommits.get(i);
            try {
                t.commit();
            } catch (RemoteException ex) {
                System.err.println("Trouble communicating: " + 
                                   ex.getMessage());
            } catch (CannotCommitException ex) {
                System.err.println("Couldn't commit: " +
                                   ex.getMessage());
            } catch (UnknownTransactionException ex) {
                System.err.println("Unknown transaction: " +
                                   ex.getMessage());
            }
        }
            
        System.out.println("Contents of space 1: ");
        dumpContents(js1);
            
        System.out.println("Contents of space 2: ");
        dumpContents(js2);
    }
    
    protected void dumpContents(JavaSpace js) {
        try {
            Name nameTmpl = new Name(null);
            
            Entry e = null;
            while ((e = js.takeIfExists(nameTmpl, null, JavaSpace.NO_WAIT)) != null) {
                System.out.println("Removing: " + e);
            }
        } catch (Exception ex) {
            System.err.println(ex.getMessage());
            ex.printStackTrace();
        }
    }
    
    public void run() {
        while (true) {
            try {
                Thread.sleep(Integer.MAX_VALUE);
            } catch (InterruptedException ex) {
            }
        }
    }
    
    public static void main(String[] args) {
        try {
            if (System.getSecurityManager() == null) {
                System.setSecurityManager(new RMISecurityManager());
            }
            EntryMover mover = new EntryMover();
            new Thread(mover).start();
        } catch (Exception ex) {
            System.err.println("Error: " + ex.getMessage());
            ex.printStackTrace();
        }
    }
}

