// Watch for DiscoveryEvents and ServiceEvents for
// a given community.  Create Entry objects for these
// and write them into a space.

package corejini.chapter17;

import net.jini.space.JavaSpace;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceEvent;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lease.Lease;
import net.jini.core.entry.Entry;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.EventRegistration;
import net.jini.discovery.DiscoveryEvent;
import net.jini.discovery.LookupDiscovery;
import net.jini.discovery.DiscoveryListener;
import net.jini.lease.LeaseRenewalManager;
import java.util.ArrayList;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import java.io.IOException;

public class DjinnWatcher implements Runnable {
    protected Class[] types = { JavaSpace.class };
    protected String group;
    protected ServiceTemplate tmpl = 
        new ServiceTemplate(null, types, null);
    protected ArrayList regs = new ArrayList();
    protected JavaSpace javaSpace = null;
    protected LeaseRenewalManager leaseManager = null;
    protected static final int ALL_FLAGS =
        ServiceRegistrar.TRANSITION_MATCH_NOMATCH |
        ServiceRegistrar.TRANSITION_NOMATCH_MATCH |
        ServiceRegistrar.TRANSITION_MATCH_MATCH;
    protected EvtListener evtListener;
    protected LookupDiscovery disco;
    
    // An inner class for listening for remote events.
    class EvtListener extends UnicastRemoteObject
        implements RemoteEventListener {
        public EvtListener() throws RemoteException {
        }
        
        public void notify(RemoteEvent ev) {
            if (!(ev instanceof ServiceEvent))
                return;
        
            ServiceEvent sev = (ServiceEvent) ev;
        
            // write the data into the space.
            writeEventData(javaSpace, sev);
        }
    }
    
    // An inner class for listening for discovery events
    class DiscListener implements DiscoveryListener {
        public void discovered(DiscoveryEvent ev) {
            boolean spaceNotYetFound = (javaSpace == null);
            ServiceRegistrar[] newregs = ev.getRegistrars();
            for (int i=0, size=newregs.length ; i<size ; i++) {
                if (!regs.contains(newregs[i])) {
                    // If we don't have a space yet, search
                    // for it.  Otherwise, start getting events
                    // from the new registrar!
                    regs.add(newregs[i]);
                
                    if (javaSpace == null) {
                        // This will register for events in
                        // all known registrars.
                        findJavaSpace(newregs[i]);
                    } else {
                        registerForEvents(newregs[i]);
                    }
                }
            }
        
            // we only log discovery events received 
            // *after* we find a space.  So if we're still
            // processing the event that found the space for
            // us, we just bail.  spaceNotYetFound records
            // whether the space was found at the time this
            // method started.
            if (spaceNotYetFound)
                return;
        
            // after adding each to the list, and looking for
            // javaspaces we can log new discovery events if
            // we've found a space already.
            if (javaSpace != null) {
                writeEventData(javaSpace, ev, true);
            }
        }
    
        public void discarded(DiscoveryEvent ev) {
            ServiceRegistrar[] deadregs = ev.getRegistrars();
            for (int i=0, size=deadregs.length ; i<size ; i++) {
                regs.remove(deadregs[i]);
            }
        
            // Log the event, if we've found a space.
            if (javaSpace != null) {
                writeEventData(javaSpace, ev, false);
            }
        }
    }

    public DjinnWatcher(String group) 
        throws IOException, RemoteException {
        leaseManager = new LeaseRenewalManager();
        evtListener = new EvtListener();
        this.group = group;

        if (group == null) { 
            System.out.println("Watching public group");  
            disco = new   
                LookupDiscovery(new String[] { "" });
        } else {  
            System.out.println("Watching group " + group);  
            disco = new  
                LookupDiscovery(new String[] { group });  
        }  
        
        disco.addDiscoveryListener(new DiscListener());  
    }

    protected void findJavaSpace(ServiceRegistrar reg) {
        if (javaSpace != null)
            return;
        
        try {
            javaSpace = (JavaSpace) reg.lookup(tmpl);
            if (javaSpace == null) {
                return;
            }
            System.out.println("Found a space!");
        } catch (RemoteException ex) {
            System.err.println("Error doing lookup: " +
                               ex.getMessage());
            disco.discard(reg);
            return;
        }
        
        for (int i=0, size=regs.size() ; i<size ; i++) {
            System.out.println("Registering for events at " +
                               "previously-found registrar: " + 
                               regs.get(i));
            registerForEvents((ServiceRegistrar) 
                              regs.get(i));
        }
    }
    
    // Log discovery events.
    protected void writeEventData(JavaSpace space, 
                                  DiscoveryEvent ev,
                                  boolean discovered) {
        System.out.println("Writing DiscoveryEvent " + ev);
        
        try {
            ServiceRegistrar[] evtregs =
                ev.getRegistrars();
            for (int i=0 ; i<evtregs.length ; i++) {
                DiscoveryEventEntry entry =
                    new DiscoveryEventEntry(group, evtregs[i],
                                            discovered);
                Lease lease = space.write(entry, null,
                                          60 * 60 * 1000);
                leaseManager.renewUntil(lease, Lease.ANY, null);
            }
        } catch (Exception ex) {
            System.err.println("Error writing event: " +
                               ex.getMessage());
        }
    }
    
    // Log service events.
    protected void writeEventData(JavaSpace space, 
                                  ServiceEvent ev) {
        System.out.println("Writing ServiceEvent " + ev);
        
        try {
            Lease lease;
            ServiceEventEntry entry = 
                new ServiceEventEntry(group, ev);
            
            lease = space.write(entry, null, 60 * 60 * 1000);
            leaseManager.renewUntil(lease, Lease.ANY, null);
        } catch (Exception ex) {
            System.err.println("Error writing event: " +
                               ex.getMessage());
        }
    }
    
    protected void registerForEvents(ServiceRegistrar reg) {
        try {
            ServiceTemplate search = 
                new ServiceTemplate(null, null, null);
            EventRegistration evtreg;
            evtreg = reg.notify(search, ALL_FLAGS, 
                                evtListener, null,
                                60 * 60 * 1000);
            leaseManager.renewUntil(evtreg.getLease(), 
                                    Lease.ANY, null);
        } catch (RemoteException ex) {
            System.err.println("Error soliciting events: " +
                               ex.getMessage());
            disco.discard(reg);
        }
    }
        
    public void run() {
        while (true) {
            try {
                Thread.sleep(Integer.MAX_VALUE);
            } catch (InterruptedException ex) {
            }
        }
    }
    
    public static void main(String[] args) {
        String group = null;
        
        if (System.getSecurityManager() == null) {
            System.setSecurityManager(new RMISecurityManager());
        }
        
        if (args.length == 1) {
            group = args[0];
        } else if (args.length > 1) {
            System.err.println("Usage: DjinnWatcher [<group>]");
        }

        try {
            DjinnWatcher dw = new DjinnWatcher(group);
            new Thread(dw).start();
            System.out.println("Cruising!");
        } catch (Exception ex) {
	    System.err.println("Error!" + ex);
        }
    }
}
