// Contact a JavaSpaces service and retrieve the
// logged history for a single group.

package corejini.chapter17;

import net.jini.space.JavaSpace;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.entry.*;
import net.jini.discovery.DiscoveryEvent;
import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.LookupDiscovery;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.io.IOException;

public class DjinnConsumer implements Runnable {
    protected Class[] types = { JavaSpace.class };
    protected String group;
    protected ServiceTemplate tmpl = 
        new ServiceTemplate(null, types, null);
    protected JavaSpace javaSpace = null;
    
    
    // An inner class for discovery.
    class DiscListener implements DiscoveryListener {
        public synchronized void discovered(DiscoveryEvent ev) {
            if (javaSpace != null)
                return;
            
            ServiceRegistrar[] regs = ev.getRegistrars();
            for (int i=0, size=regs.length ; i<size ; i++) {
                JavaSpace js = findJavaSpace(regs[i]);
                if (js != null) {
                    javaSpace = js;
                    doit();
                }
            }
        }
    
        public void discarded(DiscoveryEvent ev) {
        }
    }
    
    public DjinnConsumer(String group) throws IOException {
        LookupDiscovery disco;
        this.group = group;

        if (group == null) {
            System.out.println("Fetching public history");
            disco = new 
                LookupDiscovery(new String[] { "" });
        } else {
            System.out.println("Fetching history for group " + group);
            disco = new
                LookupDiscovery(new String[] { group });
        }
        
        disco.addDiscoveryListener(new DiscListener()); 
    }
    
    protected JavaSpace findJavaSpace(ServiceRegistrar reg) {
        if (javaSpace != null)
            return null;
        
        try {
            JavaSpace javaSpace = (JavaSpace) reg.lookup(tmpl);
            if (javaSpace != null) {
                System.out.println("Found a space!");
                return javaSpace;
            }
        } catch (RemoteException ex) {
            System.err.println("Error doing lookup: " +
                               ex.getMessage());
            return null;
        }
        
        return null;
    }
    
    protected void doit() {
        System.out.println("Searching for history for " + group);
        
        // Make a template to search for the group
        // passed on the command line, any timestamp
        WatcherEntry tmpl = new WatcherEntry();
        tmpl.group = group;
        tmpl.timestamp = null;
        
        try {
            // Snapshot it, since we'll be using it
            // over and over.
            Entry snapshot = javaSpace.snapshot(tmpl);
            boolean done = false;
            int i = 0;
            
            while (!done) {
                Entry result = javaSpace.takeIfExists(snapshot, null,
                                                      JavaSpace.NO_WAIT);
                
                if (result != null) {
                    printResult(i++, result);
                }
                
                done = (result == null);
            }
	    System.out.println("Done!");
	    System.exit(1);
        } catch (Exception ex) {
            System.err.println("bogus:  " + ex.getMessage());
            ex.printStackTrace();
        }
    }
    
    protected void printResult(int i, Entry result) {
        if (result instanceof DiscoveryEventEntry) {
            DiscoveryEventEntry disc = 
                (DiscoveryEventEntry) result;
            System.out.println("[" + i + "] " + 
                               (disc.discovered.booleanValue() ?
                                "discovered" : "discarded") +
                               " " + disc.locator.toString());
        } else if (result instanceof ServiceEventEntry) {
            ServiceEventEntry serv =
                (ServiceEventEntry) result;
            System.out.println("[" + i + "] " + serv.name + 
                               " " +  serv.transition + " " +
                               (serv.service == null ? "" :
                               serv.service.getClass().getName()) + 
                               " " + serv.serviceID);
        } else {
            // Someone else added an entry type?
            System.out.println(result);
        }
    }
    
    public void run() {
        while (true) {
            try {
                Thread.sleep(Integer.MAX_VALUE);
            } catch (InterruptedException ex) {
            }
        }
    }
    
    public static void main(String[] args) {
        String group = null;
        
        if (System.getSecurityManager() == null) {
            System.setSecurityManager(new RMISecurityManager());
        }
        
        if (args.length == 1) {
            group = args[0];
        } else if (args.length > 1) {
            System.err.println("Usage: DjinnConsumer [<group>]");
        }

        try {
            DjinnConsumer dc = new DjinnConsumer(group);
            new Thread(dc).start();
            System.out.println("Cruising!");
        } catch (Exception ex) {
	    System.err.println("Error!  " + ex);
        }
    }
}
