// A subclass of HeartbeatClient that uses a mailbox
// for deferred event delivery.

package corejini.chapter16;

import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.EventRegistration;
import net.jini.lease.LeaseRenewalManager;
import net.jini.core.lease.Lease;
import java.io.IOException;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.util.List;

public class MailboxClient extends HeartbeatClient {
    ServiceTemplate mailboxTemplate;
    EventMailbox.EventMailboxRequest mailbox = null;
    
    public MailboxClient() throws IOException, RemoteException {
        Class[] mbtypes = 
        { EventMailbox.EventMailboxRequest.class };
        mailboxTemplate = 
            new ServiceTemplate(null, mbtypes, null);
    }
    
    void doit(ServiceRegistrar reg) {
        // Bail once we've found a generator and a mailbox.
        if (gen != null && mailbox != null)
            return;
        
        // find a generator
        if (gen == null) {
            try {
                gen = (HeartbeatGenerator.HeartbeatRequest)
                    reg.lookup(template);
            } catch (Exception ex) {
                System.err.println("Doing lookup: " + ex.getMessage());
                ex.printStackTrace();
            }
        }
        
        // find a mailbox
        if (mailbox == null) {
            try {
                mailbox = (EventMailbox.EventMailboxRequest)
                    reg.lookup(mailboxTemplate);
            } catch (Exception ex) {
                System.err.println("Doing lookup: " + ex.getMessage());
                ex.printStackTrace();
            }
        }
        
        if (gen == null || mailbox == null)
            return;
        
        System.out.println("Got a generator and a mailbox!");
        
        // Remember our registrations...
        EventRegistration evtreg = null;
        EventRegistration mbreg = null;
        
        // sign us up for events!
        try {
            // Tell the generator to send to the mailbox.  Here,
            // we do our own leasing even though the events are
            // sent to the mailbox.
            evtreg =
                gen.register(HeartbeatGenerator.MINUTE,
                             null, mailbox, Lease.ANY);
            leaseManager.renewUntil(evtreg.getLease(), 
                                    Lease.ANY, null);
            
            // We need to tell the mailbox to hold on to
            // events with this <source, type>.  We do this
            // by acquiring a mailbox.  We again do our own
            // leasing of this mailbox.
            mbreg =
                mailbox.acquireMailbox(evtreg.getSource(), 
                                       evtreg.getID(),
                                       Lease.ANY);
            leaseManager.renewUntil(mbreg.getLease(), 
                                    Lease.ANY, null);
        } catch (RemoteException ex) {
            System.err.println("Bogus: " + ex.getMessage());
            ex.printStackTrace();
        }
        
        System.out.println("Sleeping a while...");
        
        // now wait a while, then ask the mailbox for our events.
        try {
            Thread.sleep(1000 * 60 * 5);
        } catch (InterruptedException ex) {
        }
        
        System.out.println("Getting events in a batch");
        
        try {
            List events = mailbox.getEvents(mbreg.getSource(),
                                              mbreg.getID());
            System.out.println("events:" );
            for (int i=0, size=events.size() ; i<size ; i++) {
                System.out.println(i + " = " + events.get(i));
            }
            // or... mailbox.deliverEvents()?  uses notify()...
        } catch (Exception ex) {
            ex.getMessage();
            ex.printStackTrace();
        }
            
        System.out.println("Sleeping again...");
        
        // now wait a while, then ask the mailbox for our events.
        try {
            Thread.sleep(1000 * 60 * 5); 
        } catch (InterruptedException ex) {
        }
        
        System.out.println("Asking for delivery");
        
        try {
            mailbox.deliverEvents(mbreg.getSource(), 
                                  mbreg.getID(),
                                  listener);
        } catch (Exception ex) {
            ex.getMessage();
            ex.printStackTrace();
        }
    }
    
    public static void main(String[] args) {
	if (System.getSecurityManager() == null) {
	    System.setSecurityManager(
		new RMISecurityManager());
	}

        try {
            MailboxClient client = new MailboxClient();
            new Thread(client).start();
        } catch (Exception ex) {
            System.err.println("Bogus: " + ex.getMessage());
            System.exit(1);
        }
    }
}


