//  A simple service that sends periodic events.

package corejini.chapter16;

import corejini.chapter15.BasicUnicastService;
import corejini.chapter15.BasicUnicastAdmin;
import net.jini.core.lease.Lease;
import net.jini.core.lease.UnknownLeaseException;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.EventRegistration;
import net.jini.core.event.UnknownEventException;
import com.sun.jini.lease.landlord.LandlordLease;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.MarshalledObject;
import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import java.io.Serializable;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;

// The cookie class for our leases.  This class
// identifies leases by their unique event type
// identifiers and the period they're registering
// for.
class HeartbeatCookie implements Serializable {
    int period;
    long eventType;
    
    HeartbeatCookie(int period, long eventType) {
        this.period = period;
        this.eventType = eventType;
    }
    public boolean equals(Object other) {
        if (!(other instanceof HeartbeatCookie))
            return false;
        
        HeartbeatCookie cookie = (HeartbeatCookie) other;
        
        return cookie.period == period && cookie.eventType == eventType;
    }
}

// The heartbeat service is a BasicUnicastService
// (see Chapter 13).
public class HeartbeatGenerator 
        extends BasicUnicastService
        implements Runnable, HeartbeatGenerator.HeartbeatRequest {
    // 10 minutes
    protected static final int MAX_LEASE = 1000 * 60 * 10; 
    protected boolean done = false;
    protected long tickCount = 0;
    // Lists of registrants.
    protected List minute = new ArrayList();
    protected List hour = new ArrayList();
    protected List day = new ArrayList();
    protected List week = new ArrayList();
    // Separate landlords for each list.
    protected BasicRegistrationLandlord minuteLandlord;
    protected BasicRegistrationLandlord hourLandlord;
    protected BasicRegistrationLandlord dayLandlord;
    protected BasicRegistrationLandlord weekLandlord;
    protected long lastEventSeqNo = 0;
    protected LandlordLease.Factory factory = 
        new LandlordLease.Factory();
    
   protected long nextEventType = 1;

    public static final int MINUTE=1, HOUR=2, DAY=3, WEEK=4;
    
    // Saves our class-specific data when we persist
    static class Registrations implements Serializable {
        List minute;
        List hour;
        List day;
        List week;
        long nextEventType;
        long lastEventSeqNo;
        
        Registrations(List minute, List hour, 
                      List day, List week, 
                      long nextEventType, long lastEventSeqNo) {
            this.minute = minute;
            this.hour = hour;
            this.day = day;
            this.week = week;
            this.nextEventType = nextEventType;
            this.lastEventSeqNo = lastEventSeqNo;
        }
    }
    
    // We send particular subclasses of RemoteEvent
    public static class HeartbeatEvent extends RemoteEvent 
        implements Serializable {
        int period;
        
        HeartbeatEvent(Object proxy, long eventType, long seq, 
                       MarshalledObject data, int period) {
            super(proxy, eventType, seq, data);
            this.period = period;
        }
        public int getPeriod() {
            return period;
        }
    }
    
    // Callers speak this interface to ask us to send heartbeats
    public interface HeartbeatRequest extends Remote {
        public EventRegistration register(int period,
                                          MarshalledObject data,
                                          RemoteEventListener listener,
                                          long duration)
            throws RemoteException;
    }
    
    public HeartbeatGenerator(String storageLoc) throws RemoteException {
        super(storageLoc);
        
        minuteLandlord = new BasicRegistrationLandlord(minute, factory);
        hourLandlord = new BasicRegistrationLandlord(hour, factory);
        dayLandlord = new BasicRegistrationLandlord(day, factory);
        weekLandlord = new BasicRegistrationLandlord(week, factory);
    }
   
    // Initialize the superclass and start the
    // leasing thread.
    protected void initialize() throws IOException, ClassNotFoundException {
        super.initialize();
        new Thread(this).start();
    }
   
    // Save the specific registration data
    protected void checkpoint() throws IOException {
        checkpoint(new Registrations(minute, hour, day, week,
                                     nextEventType, lastEventSeqNo));
    }

    // Use the RMI-generated stub as the proxy
    protected Object getProxy() {
        return this;
    }

    // Restore class-specific data.
    public void restored(Object subclassData) {
        Registrations regs = (Registrations) subclassData;
        minute = regs.minute;
        hour = regs.hour;
        day = regs.day;
        week = regs.week;
        nextEventType = regs.nextEventType;
        lastEventSeqNo = regs.lastEventSeqNo;
    }
    
    public synchronized EventRegistration register(int period,
                                      MarshalledObject data,
                                      RemoteEventListener listener,
                                      long duration) {
        // Build a cookie based on the table and token
	long eventType = nextEventType++;
        HeartbeatCookie cookie = new HeartbeatCookie(period, eventType);
        
        // The landlord we'll use for the lease.
        BasicRegistrationLandlord landlord;
        
        // The registration list we'll add it to.
        List regs;
        
        switch (period) {
        case MINUTE:
            landlord = minuteLandlord;
            regs = minute;
            break;
        case HOUR:
            landlord = hourLandlord;
            regs = hour;
            break;
        case DAY:
            landlord = dayLandlord;
            regs = day;
            break;
        case WEEK:
            landlord = weekLandlord;
            regs = week;
            break;
        default:
            throw new IllegalArgumentException("Bad period");
        }
        
        // Create the lease and registration, and add the
        // registration to the appropriate list.
        long expiration = landlord.getExpiration(duration);
        Registration reg = new Registration(cookie, listener,
                                            data, expiration);
        Lease lease = factory.newLease(cookie, landlord, expiration);
        regs.add(reg);
        
        // Return an event registration to the client.
        EventRegistration evtreg = 
            new EventRegistration(eventType, proxy, lease, 0);
        
        try {
            checkpoint();
        } catch(IOException ex) {
            System.err.println("Error checkpointing: " + 
                               ex.getMessage());
        }
        
        return evtreg;
    }
    
    protected void sendHeartbeat() {
        tickCount++;
        
        // minute listeners
        sendHeartbeats(minute, MINUTE);
        
        // hour listeners
        if (tickCount % 60 == 0) {
            sendHeartbeats(hour, HOUR);
        }
        
        // day listeners
        if (tickCount % (60 * 24) == 0) {
            sendHeartbeats(day, DAY);
        }
        
        // week listeners
        if (tickCount % (60 * 24 * 7) == 0) {
            sendHeartbeats(week, WEEK);
        }
    }
    
    protected void sendHeartbeats(List regs, int period) {
        // First, scavenge the list for dead registrations, in
        // reverse order (to make us immune from compaction)
        long now = System.currentTimeMillis();
        for (int i=regs.size()-1 ; i >= 0 ; i--) {
            Registration reg = (Registration) regs.get(i);
            if (reg.expiration < now) {
                reg.cancelled();
                regs.remove(i);
            }
        }
        
        // Now, message the remaining listeners
        for (int i=0, size=regs.size() ; i<size ; i++) {
            Registration reg = (Registration) regs.get(i);
	    HeartbeatCookie cookie = 
		(HeartbeatCookie) reg.getCookie();
	    long eventType = cookie.eventType;
            
            try {
                HeartbeatEvent ev = new HeartbeatEvent(proxy, 
					               eventType,	
                                                       lastEventSeqNo++,
                                                       reg.data, period);
                reg.listener.notify(ev);
            } catch (RemoteException ex) {
		// Just complain...
                System.err.println("Error notifying remote listener: " +
                                   ex.getMessage());
            } catch (UnknownEventException ex) {
		// Cancel the registration... Here I'll
		// do this by setting its expiration so
		// that it'll be dropped the next time
		// through.
                System.err.println("Unknown event, dropping: " +
                                   ex.getMessage());
		reg.expiration = 0;
            }
        }
    }
    
    public void run() {
        long timeToSleep = 60 * 1000;
        while (true) {
            long nextWakeup = System.currentTimeMillis() + timeToSleep;
            try {
                Thread.sleep(timeToSleep); 
            } catch (InterruptedException ex) {
            }
            long currentTime = System.currentTimeMillis();
            // see if we're at the next wakeup time
            if (currentTime >= nextWakeup) {
                nextWakeup = currentTime + (60 * 1000);
                // notify
                sendHeartbeat();
            }
            timeToSleep = nextWakeup - System.currentTimeMillis();
        }
    }
    
    public static void main(String[] args) {
	if (System.getSecurityManager() == null) {
	    System.setSecurityManager(
		new RMISecurityManager());
	}

        try {
            if (args.length != 1) {
                System.err.println("Usage: HeartbeatGenerator " +
                                   "<storageloc>");
                System.exit(1);
            }
            
            HeartbeatGenerator gen = new HeartbeatGenerator(args[0]);
            gen.initialize();
            System.out.println("Heartbeat generator started.");
        } catch (Exception ex) {
            System.err.println("Error starting heartbeat generator: " +
                               ex.getMessage());
            System.exit(1);
        }
    }
}
