// A "mailbox" for events--a service that stores
// events until asked.

package corejini.chapter16;

import corejini.chapter15.BasicUnicastService;
import corejini.chapter15.BasicUnicastAdmin;
import net.jini.core.lease.Lease;
import net.jini.core.lease.UnknownLeaseException;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.EventRegistration;
import net.jini.core.event.UnknownEventException;
import com.sun.jini.lease.landlord.Landlord;
import com.sun.jini.lease.landlord.LandlordLease;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import java.io.IOException;
import java.io.Serializable;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;

// Define a new type of cookie to use for our
// leases.
class MailboxCookie implements Serializable {
    protected Object source;
    protected long eventType;
    
    MailboxCookie(Object source, long eventType) {
        this.source = source;
        this.eventType = eventType;
    }
    
    public boolean equals(Object o) {
        if (!(o instanceof MailboxCookie))
            return false;
        
        MailboxCookie cookie = (MailboxCookie) o;
        
        return cookie.eventType == eventType &&
            cookie.source.equals(source);
    }
}

// EventMailbox is a BasicUnicastService (see Ch. 13)
public class EventMailbox extends BasicUnicastService 
    implements Runnable, EventMailbox.EventMailboxRequest {
    protected LandlordLease.Factory factory =
        new LandlordLease.Factory();
    // This is a hashmap keyed by 'source'.  Each
    // value is a list of Registrations.
    protected HashMap mailboxes = new HashMap();
    // This is a hashmap of all of the landlords
    // for the individual registration lists; it
    // is keyed by 'source'.
    protected HashMap landlords = new HashMap();
    
    // Defines the remote interface that clients will
    // use to communicate with the service.
    public interface EventMailboxRequest 
        extends RemoteEventListener, Remote {
        // Get a mailbox.
        public EventRegistration acquireMailbox(Object source,
                                                long eventType,
                                                long duration)
            throws RemoteException;
        // Get events in a batch.
        public List getEvents(Object source, 
                              long eventType) 
            throws RemoteException, 
                    UnknownLeaseException;
        // Have events re-delivered to the original
        // listener.
        public void deliverEvents(Object source, long eventType, 
                                  RemoteEventListener listener)
            throws RemoteException, UnknownLeaseException;
    }
    
    // serialized data object
    static class MailboxData implements Serializable {
        HashMap mailboxes;
        
        MailboxData(HashMap mailboxes) {
            this.mailboxes = mailboxes;
        }
    }
    
    protected EventMailbox(String storageLoc) throws RemoteException {
        super(storageLoc);
    }
    
    public void initialize() throws IOException, ClassNotFoundException {
        super.initialize();
        new Thread(this).start();
    }
    protected void checkpoint() throws IOException {
        super.checkpoint(new MailboxData(mailboxes));
    }
    protected Object getProxy() {
        return this;
    }
    public void restored(Object subclassData) {
        MailboxData data = (MailboxData) subclassData;
        mailboxes = data.mailboxes;
    }
    
    //
    // clients, when they come in, must have the correct key token in
    // their cookies, or we assume that they're holding on to an
    // expired cookie.  the mailboxcookie.equals() method checks for
    // token equality.  but when events come in from generators, we
    // still have to find where to stash them--which mailbox--but we
    // don't have a token.  so we "manually" scan the list looking for
    // any registration with matching source and type, but ignoring the
    // key.
    
    // Events from generators come in through this method.
    public synchronized void notify(RemoteEvent ev) {
        Object source = ev.getSource();
        List regs = (List) mailboxes.get(source);
        
        // If we don't have an entry for this source, we
        // don't complain with an UnknownEventException--
        // that would prevent us from getting any future
        // events from that source.  Instead we just
        // silently ignore it.
        if (regs == null) {
            System.err.println("Unknown source.");
            return;
        }
        
        // Iterate through the list looking for the
        // matching cookie.
        for (int i=0, size=regs.size() ; i<size ; i++) {
            EventMailboxRegistration reg = 
                (EventMailboxRegistration) regs.get(i);
            
            MailboxCookie cookie = (MailboxCookie) reg.getCookie();
            
            // File the event into the matching mailbox.
            if (cookie.source.equals(source)) {
                System.out.println("Saved event in mailbox");
                reg.appendEvent(ev);
                return;
            }
        }
        
        // If we don't find a matching mailbox, just ignore it.
        System.out.println("No registration.");
    }
    
    // Consumers or generators can install a request for
    // mailboxing through this call.
    public EventRegistration acquireMailbox(Object source,
                                            long eventType,
                                            long duration) 
        throws RemoteException {
        // Create a cookie for the mailbox and its lease
        MailboxCookie cookie = new MailboxCookie(source, eventType);
        
        // Create a new list for this source, if we
        // need to.  Also get the landlord for this
        // source.
        List regs;
        BasicRegistrationLandlord landlord;
        if ((regs = (List) mailboxes.get(source)) == null) {
            regs = new ArrayList();
            landlord = new BasicRegistrationLandlord(regs, factory);
            mailboxes.put(source, regs);
            landlords.put(source, landlord);
        } else {
            landlord = (BasicRegistrationLandlord) 
                landlords.get(source);
        }
        
        // Compute the expiration time.
        long expiration = landlord.getExpiration(duration);
        EventMailboxRegistration emr = new
            EventMailboxRegistration(cookie, null, null,
                                     expiration);
        Lease lease = factory.newLease(cookie, landlord, expiration);
        
        // Add the registration to the list of mailboxes.
        regs.add(emr);
        
        // Return an event registration to the client
        EventRegistration mbr = 
            new EventRegistration(eventType, source, lease, 0);
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Error checkpoint: " + 
                               ex.getMessage());
        }
        
        System.out.println("Acquired new mailbox for " +
                           source + ", " + eventType);
        
        return mbr;
    }

    public synchronized List getEvents(Object source,
                                       long eventType) 
        throws UnknownLeaseException {
        System.out.println("Getting events.");
        List regs = (List) mailboxes.get(source);
        
        if (regs == null) {
            throw new UnknownLeaseException();
        }
        
        MailboxCookie cookie = new MailboxCookie(source, 
                                                 eventType);
        
        for (int i=0, size=regs.size() ; i<size ; i++) {
            EventMailboxRegistration reg = 
                (EventMailboxRegistration) regs.get(i);
            if (reg.getCookie().equals(cookie))
                return reg.getAndFlushMailbox();
        }
        
        throw new UnknownLeaseException();
    }

    public synchronized void deliverEvents(Object source, 
                                           long eventType,
                                           RemoteEventListener listener) 
        throws UnknownLeaseException, RemoteException {
        System.out.println("Redelivering events.");
        List regs = (List) mailboxes.get(source);
        
        if (regs == null) {
            throw new UnknownLeaseException();
        }
        
        MailboxCookie cookie = new MailboxCookie(source,
                                                 eventType);
        
        for (int i=0, size=regs.size() ; i<size ; i++) {
            EventMailboxRegistration reg = 
                (EventMailboxRegistration) regs.get(i);
            
            if (reg.getCookie().equals(cookie)) {
                List events = reg.getMailbox();
                // Try to send them.  If we get any exceptions, don't
                // flush the mailbox.
                boolean succeeded = true;
                for (int j=0, evsize=events.size() ; j<evsize ; j++) {
                    try {
                        listener.notify((RemoteEvent) 
                                        events.get(j));
                    } catch (RemoteException ex) {
                        System.err.println("Error sending: " + 
                                           ex.getMessage());
                        succeeded = false;
                    } catch (UnknownEventException ex) {
                        System.err.println(
                           "Client claims they don't want " +
                           "this event type... removing them");
                        
                        Landlord ll = (Landlord)
                            landlords.get(source);
                        
                        if (ll != null) {
                            ll.cancel(reg.getCookie());
                        }
                        
                        return;
                    }
                }
                
                if (succeeded) {
                    reg.getAndFlushMailbox();
                }
                
                return;
            }
        }
        
        throw new UnknownLeaseException();
    }
    
    public void run() {
        while (true) {
            try {
                Thread.sleep(Integer.MAX_VALUE);
            } catch (InterruptedException ex) {
            }
        }
    }
    
    public static void main(String[] args) {
	if (System.getSecurityManager() == null) {
	    System.setSecurityManager(
		new RMISecurityManager());
	}

        if (args.length != 1) {
            System.err.println("Usage: EventMailbox <storage>");
            System.exit(1);
        }
        
        try {
            EventMailbox mailbox = new EventMailbox(args[0]);
            mailbox.initialize();
            System.out.println("running!");
        } catch (Exception ex) {
            System.err.println("Error starting event mailbox: " +
                               ex.getMessage());
            System.exit(1);
        }
    }
}

