// This class implements some basic functionality to
// lease event registrations.  It can work in 
// conjunction with the BasicUnicastService from
// Chapter 14.

package corejini.chapter16;

import net.jini.core.lease.Lease;
import net.jini.core.lease.UnknownLeaseException;
import net.jini.core.lease.LeaseDeniedException;
import com.sun.jini.lease.landlord.Landlord;
import com.sun.jini.lease.landlord.LandlordLease;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;

public class BasicRegistrationLandlord 
	extends UnicastRemoteObject
    	implements Landlord {
    // A simple leasing policy...10 minute leases.
    protected static final int DEFAULT_MAX_LEASE 
	= 1000 * 60 * 10;
    protected int maxLease = DEFAULT_MAX_LEASE;
    // Assume that registrations are kept in
    // a list maintained by our clients.
    protected List regs;
    // A factory for making landlord leases.
    protected LandlordLease.Factory factory;
    
    public BasicRegistrationLandlord(List regs, 
                                     LandlordLease.Factory factory)
        throws RemoteException {
        this.regs = regs;
        this.factory = factory;
    }
   
    // Change the maximum lease time from the default.
    public void setMaxLease(int maxLease) {
        this.maxLease = maxLease;
    }

    // Apply the policy to a requested duration
    // to get an actual expiration time.
    public long getExpiration(long request) {
        if (request > maxLease || request == Lease.ANY)
            return System.currentTimeMillis() + maxLease;
        else
            return System.currentTimeMillis() + request;
    }
   
    // Cancel the lease represented by 'cookie'
    public void cancel(Object cookie) 
        throws UnknownLeaseException, RemoteException {
        for (int i=0, size=regs.size() ; i<size ; i++) {
            Registration reg = 
		(Registration) regs.get(i);
            if (reg.cookie.equals(cookie)) {
                reg.cancelled();
                regs.remove(i);
                return;
            }
        }
        throw new UnknownLeaseException(cookie.toString());
    }

    // Cancel a set of leases
    public Map cancelAll(Object[] cookies) 
        throws RemoteException {
        Map exceptionMap = null;
        
        for (int i=0 ; i<cookies.length ; i++) {
            try {
                cancel(cookies[i]);
            } catch (UnknownLeaseException ex) {
                if (exceptionMap == null) {
                    exceptionMap = new HashMap();
                    exceptionMap.put(cookies[i], ex);
                }
            }
        }
        
        return exceptionMap;
    }

    // Renew the lease specified by 'cookie'
    public long renew(Object cookie, long extension)
        throws UnknownLeaseException, LeaseDeniedException, RemoteException {
        
        for (int i=0, size=regs.size() ; i<size ; i++) {
            Registration reg = 
			(Registration) regs.get(i);
            if (reg.getCookie().equals(cookie)) {
                long expiration = 
			getExpiration(extension);
                reg.setExpiration(expiration);
                return expiration - 
			System.currentTimeMillis();
            }
        }
        throw new UnknownLeaseException(cookie.toString());
    }

    // Renew a set of leases.
    public Landlord.RenewResults renewAll(Object[] cookies,
                                          long[] extensions) 
            throws RemoteException { 
        long[] granted = new long[cookies.length];
        Exception[] denied = null;
        
        for (int i=0 ; i<cookies.length ; i++) {
            try {
                granted[i] = renew(cookies[i], 
				   extensions[i]);
            } catch (Exception ex) {
                if (denied == null) {
                    denied = new Exception[cookies.length+1];
                }
                denied[i+1] = ex;
            }
        }
            
        Landlord.RenewResults results =
            new Landlord.RenewResults(granted, denied);
        return results;
    }
}
