// This is the print service back-end.

package corejini.chapter15;

import net.jini.core.entry.Entry;
import net.jini.core.event.RemoteEventListener;
import net.jini.admin.JoinAdmin;
import com.sun.jini.admin.DestroyAdmin;
import com.sun.jini.admin.StorageLocationAdmin;
import net.jini.lookup.entry.ServiceInfo;
import java.awt.print.Printable;
import java.awt.print.Pageable;
import java.awt.print.PageFormat;
import java.awt.print.PrinterJob;
import java.io.Serializable;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;

public class PrintService extends BasicUnicastService
    implements RemotePrinter {
    protected List queue = new ArrayList();
    protected HashMap agents = new HashMap();
    protected PrintThread printThread;
    protected boolean done = false;
    protected PrintRecord currentRecord = null;
    protected PrintAdmin remoteAdmin = null;
    protected long lastEventSeqNo = 0;

    // An implementation of RemotePrintAdmin, using the
    // administration toolkit.

    class PrintAdminImpl extends BasicUnicastAdmin
        implements RemotePrintAdmin {

        PrintAdminImpl(PrintService s) throws RemoteException {
            super(s);
        }

        public List getJobs() {
            List jobs = new ArrayList();
            synchronized (PrintService.this) {
                if (currentRecord != null) {
                    jobs.add(new JobRecord(currentRecord, true));
                }

                for (int i=0, size=queue.size() ; i<size ; i++) {
                    jobs.add(new JobRecord((PrintRecord)
                                           queue.get(i),
                                           false));
                }
            }
            return jobs;
        }
    }


    public PrintService(String storageLoc) throws RemoteException {
        super(storageLoc);
    }
    
    // overrides base implementation to get 
    // correct shutdown behavior
    protected void shutdown() {
        done = true;
        notify();
        super.shutdown();
    }
    
    // overrides base implementation to also 
    // save PrintServiceData
    protected void checkpoint() throws IOException {
        checkpoint(new PrintServiceData(currentRecord, queue,
                                        lastEventSeqNo));
    }
    
    // overrides base implementation to start
    // print thread
    protected void initialize() throws IOException, ClassNotFoundException {
        super.initialize();
        
        printThread = new PrintThread();
        printThread.start();
    }

    // overrides base implementation to set some
    // attributes.
    protected Entry[] getAttributes() {
        Entry[] entries = new Entry[2];
        entries[0] = new ServiceInfo("Prentice-Hall",
                                     "PrintService",
                                     "PrintService",
                                     null,
                                     "Prentice-Hall",
                                     "v1.0");
        entries[1] = new PrintAdminPanel();
        return entries;
    }
    
    // overrides base implementation to return
    // print proxy.
    protected Object getProxy() {
        return new PrintProxy(this);
    }
   
    // overrides base implementation to restore
    // print service-specific data.
    public void restored(Object subclassData) {
        PrintServiceData data = (PrintServiceData) subclassData;
        currentRecord = data.currentRecord;
        queue = data.queue;
        lastEventSeqNo = data.lastEventSeqNo;
        System.out.println("Restored!");
    }
    
    public void print(int format, int copies, Object data, 
                      RemoteEventListener listener) {
        Object agent = getPrintAgent(data);
        PrintRecord record = new PrintRecord(format, copies, data,
                                             agent, listener);
        queuePrinting(record);
    }
    
    public Object getAdmin() throws RemoteException {
        if (remoteAdmin == null) {
            remoteAdmin = new PrintAdminImpl(this);
        }
            
        return remoteAdmin;
    }
    
    Object getPrintAgent(Object data) {
        if (data instanceof Printable) {
            System.out.println("Data is printable");
            return data;
        }
        if (data instanceof Pageable) {
            System.out.println("Data is pageable");
            return data;
        }
        
        // build a list of data's classes and search for
        // them, most specific to least.
        List classes = getClasses(data);
        
        for (int i=0, size=classes.size() ; i<size ; i++) {
            Object o = agents.get(classes.get(i));
            if (o != null) {
                System.out.println("Found printer for " +
                                   ((Class) classes.get(i)).getName());
                return o;
            }
        }
        
        System.out.println("No printer found");
        return null;
    }
    
    List getClasses(Object data) {
        List classes = new ArrayList();
        Class nextClass = data.getClass();
        
        while (nextClass != null) {
            classes.add(nextClass);
            Class[] interfaces = nextClass.getInterfaces();
            for (int i=0, size=interfaces.length ; i<size ; i++) {
                classes.add(interfaces[i]);
            }
            
            nextClass = nextClass.getSuperclass();
        }
        
        return classes;
    }
    
    void queuePrinting(PrintRecord record) {
        queue.add(record);
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Trouble checkpointing; " +
                               "couldn't save job: " +
                               ex.getMessage());
        }
        System.out.println("Printing queueued, waking thread");
        
        synchronized(this) {
            notify();
        }
    }

    // PrintThread handles the actual mechanics of job management
    // and printing
    class PrintThread extends Thread {
        public void run() {
            System.out.println("PrintThread cranking...");
            while (!done) {
                PrintRecord record = null;
                
                System.out.println("Waiting for lock to dequeue");
                //
                // Grab the next item off the queue.  Check
                // currentRecord, since we may be initialized
                // with a print job in-progress after a restore.
                //
                if (currentRecord == null) {
                    synchronized (PrintService.this) {
                        System.out.println("Dequeueing...");
                        if (queue.size() != 0) {
                            record = (PrintRecord) queue.get(0);
                            currentRecord = record;
                            queue.remove(0);
                            try {
                                checkpoint();
                            } catch (IOException ex) {
                                System.out.println("Couldn't checkpoint; " +
                                                   "job not saved: " +
                                                   ex.getMessage());
                            }
                        }
                    }
                } else {
                    record = currentRecord;
                }
                
                //
                // Print it, if we have something to print.
                //
                if (record != null) {
                    Exception ex = printRecord(record);
                    if (record.listener != null) {
                        PrintServiceEvent ev = 
                            new PrintServiceEvent((PrintProxy) proxy, 
                                                  lastEventSeqNo++, 
                                                  ex);
                        System.out.println("Sending event from proxy " +
                                           proxy);
                        try {
                            record.listener.notify(ev);
                        } catch (Exception ex2) {
                            System.err.println("Trouble sending event: " +
                                               ex2.getMessage());
                        }
                    }
                    currentRecord = null;
                    try {
                        checkpoint();
                    } catch (IOException ex2) {
                        System.out.println("Couldn't checkpoint; " +
                                           "job not saved: " +
                                           ex2.getMessage());
                    }
                }
                
                //
                // Wait for more stuff on the queue...
                //
                if (queue.size() > 0)
                    continue;
                
                System.out.println("Waiting for lock to wait");
                synchronized(PrintService.this) {
                    try {
                        System.out.println("waiting...");
                        PrintService.this.wait();
                    } catch (InterruptedException ex) {
                        System.out.println("interrupted...");
                    }
                }
                
                System.out.println("Waking up!  Something to print!");
            }
            System.out.println("Shutting down print thread...");
        }
        
        Exception printRecord(PrintRecord record) {
            Exception ex = null;
            System.out.println("Printing data " + record.data);
            
            PageFormat format = new PageFormat();
            format.setOrientation(record.format);
            
            PrinterJob job = PrinterJob.getPrinterJob();
            job.setCopies(record.copies);
            job.defaultPage(format);
            
            if (record.agent instanceof Pageable) {
                job.setPageable((Pageable) record.agent);
                try {
                    job.print();
                } catch (Exception ex2) {
                    ex = ex2;
                }
            } else if (record.agent instanceof Printable) {
                job.setPrintable((Printable) record.agent, format);
                try {
                    job.print();
                } catch (Exception ex2) {
                    ex2.printStackTrace();  /// WKE
                    ex = ex2;
                }
            } else {
                ex = new IllegalArgumentException("Bad agent");
            }
            
            return ex;
        }
    }
    
    public static void main(String[] args) {
        try {
            if (args.length != 1) {
                System.err.println("Usage: PrintService <datafile>");
                System.exit(1);
            }
            
            if (System.getSecurityManager() == null) {
                System.setSecurityManager(new RMISecurityManager());
            }
        
            PrintService service = new PrintService(args[0]);
            service.initialize();
            System.out.println("Print Service Started");
        } catch (Exception ex) {
            System.err.println("Error starting print service: " +
                               ex.getMessage());
            System.exit(1);
        }
    }
}

