// A client of the print service

package corejini.chapter15;

import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.DiscoveryEvent;
import net.jini.discovery.LookupDiscovery;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceMatches;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.entry.Entry;
import net.jini.admin.Administrable;
import java.io.IOException;
import java.io.Serializable;
import java.awt.Graphics;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PageFormat;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import javax.swing.JFrame;

public class Client implements Runnable { 
    ServiceTemplate template;

    // An inner class for discovery
    class Discoverer implements DiscoveryListener {
        public void discovered(DiscoveryEvent ev) {
            ServiceRegistrar[] regs = ev.getRegistrars();
            for (int i=0 ; i<regs.length ; i++) {
                doit(regs[i]);
            }
        }
        public void discarded(DiscoveryEvent ev) {
        }
    }
    
    public Client() throws IOException {
        Class[] types = { Printer.class };
        template = new ServiceTemplate(null, types, null);
        
        LookupDiscovery disco =
            new LookupDiscovery(LookupDiscovery.ALL_GROUPS);
        disco.addDiscoveryListener(new Discoverer());
    }

    void doit(ServiceRegistrar reg) {
        Printer printer = null;
        Entry[] attributes = null;
        
        try {
            ServiceMatches matches = reg.lookup(template, 
                                                Integer.MAX_VALUE);
            if (matches.totalMatches > 0) {
                printer = (Printer) matches.items[0].service;
                attributes = matches.items[0].attributeSets;
            } else {
                printer = null;
            }
        } catch (Exception ex) {
            System.err.println("Doing lookup: " + ex.getMessage());
            ex.printStackTrace();
        }
        
        if (printer == null)
            return;
        
        System.out.println("Got a printer!");
        
        // issue a couple of print requests.
        try {
            printer.print(PageFormat.PORTRAIT, 1, 
                          new PrintableString("Hello World"), 
                          new Listener());
            printer.print(PageFormat.LANDSCAPE, 1, 
                          new PrintableString("Hello Again"), 
                          new Listener());
        } catch (Exception ex) {
            System.out.println("Trouble printing: " + ex.getMessage());
            ex.printStackTrace();
        }
        
        PrintAdminPanel panel = null;
        
        for (int i=0, size=attributes.length ; i<size ; i++) {
            if (attributes[i] instanceof PrintAdminPanel) 
                panel = (PrintAdminPanel) attributes[i];
		break;
        }

        if (panel != null) {
            try {
                panel.setAdmin((PrintAdmin) 
                               ((Administrable) printer).getAdmin());
            } catch (RemoteException ex) {
                System.err.println("Error getting admin: " + ex.getMessage());
                ex.printStackTrace();
            }
        
            final JFrame frame = new JFrame("Print Admin Panel");
            frame.getContentPane().add(panel.getPanel());
            frame.pack();
            frame.setVisible(true);
            frame.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent ev) {
                    frame.setVisible(false);
                    frame.dispose();
                }
            });
        }
    }
   
    // An inner class for handling events.
    class Listener extends UnicastRemoteObject implements RemoteEventListener {
        Listener() throws RemoteException {
            super();
        }
        public void notify(RemoteEvent e) {
            if (!(e instanceof PrintServiceEvent)) {
                return;
            }
            PrintServiceEvent ev = (PrintServiceEvent) e;
            if (ev.printJobFinished()) {
                System.out.println("Job finished successfully!");
            } else {
                System.out.println("Job failed:  " + 
                                   ev.getException().getMessage());
                ev.getException().printStackTrace();
            }
        }
    }
    
    public void run() {
        while (true) {
            try {
                Thread.sleep(Integer.MAX_VALUE);
            } catch (InterruptedException ex) {
            }
        }
    }
   
    // A string that implements Printable
    static class PrintableString implements Printable, Serializable {
        String s;
        PrintableString(String s) {
            this.s = s;
        }
        public int print(Graphics g, PageFormat pf, int page) throws
            PrinterException {
            if (page >= 1) {
                return Printable.NO_SUCH_PAGE;
            }
            int xo = (int) pf.getImageableX();
            int yo = (int) pf.getImageableY();
	    g.setColor(java.awt.Color.black);
            g.drawString(s, xo+20, yo+60);
            return Printable.PAGE_EXISTS;
        }
        public String toString() { 
            return s;
        }
    }

    public static void main(String[] args) {
        try {
            if (System.getSecurityManager() == null) {
                System.setSecurityManager(new RMISecurityManager());
            }
            
            Client client = new Client();
            new Thread(client).start();
        } catch (Exception ex) {
            System.err.println("Bogus: " + ex.getMessage());
            System.exit(1);
        }
    }
}
