// A class to use as a superclass for RMI-based services.

package corejini.chapter15;

import net.jini.core.lookup.ServiceID;
import net.jini.core.entry.Entry;
import net.jini.core.discovery.LookupLocator;
import net.jini.lookup.JoinManager;
import net.jini.lookup.ServiceIDListener;
import net.jini.lease.LeaseRenewalManager;
import net.jini.discovery.LookupDiscoveryManager;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.io.File;
import java.io.Serializable;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.FileOutputStream;
import java.io.FileInputStream;

public abstract class BasicUnicastService 
        extends UnicastRemoteObject 
        implements ServiceIDListener {
    protected JoinManager joinMgr;
    protected LeaseRenewalManager leaseMgr;
    protected LookupDiscoveryManager luMgr;
    protected String storageLoc;
    protected File file = null;
    protected ServiceID serviceID = null;
    protected Object proxy = null;

    // Store data for crash recovery
    static class PersistentData implements Serializable {
        ServiceID serviceID;
        Entry[] attrs;
        String[] groups;
        LookupLocator[] locs;
        Object subclassData;
        
        PersistentData(ServiceID serviceID, Entry[] attrs, String[] groups,
                       LookupLocator[] locs, Object subclassData) {
            this.serviceID = serviceID;
            this.attrs = attrs;
            this.groups = groups;
            this.locs = locs;
            this.subclassData = subclassData;
        }
    }
    
    public BasicUnicastService(String storageLoc) throws RemoteException {
        super();
        
        this.storageLoc = storageLoc;
        file = new File(storageLoc);
    }

    // subclasses that want to be informed after a
    // restore should override this.
    public void restored(Object subclassData) {
    }
    
    public JoinManager getJoinManager() {
        return joinMgr;
    }
    
    public LookupDiscoveryManager getDiscoveryManager() {
        return luMgr;
    }
    
    public LeaseRenewalManager getLeaseManager() {
        return leaseMgr;
    }
    
    // administration will call this to shutdown.  clients can override.
    protected void shutdown() {
        System.exit(1);
    }
    
    protected abstract Object getProxy();
    
    // subclasses override to return initial attributes
    protected Entry[] getAttributes() {
        return new Entry[0];
    }
    
    public void serviceIDNotify(ServiceID id) {
        serviceID = id;
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Trouble checkpointing: " +
                               ex.getMessage());
        }
    }
    
    protected void checkpoint() throws IOException {
        checkpoint(null);
    }
    protected void checkpoint(Object subclassData) throws IOException {
        PersistentData data;
        
        data = new PersistentData(serviceID,
                                  joinMgr.getAttributes(),
                                  luMgr.getGroups(),
                                  luMgr.getLocators(),
                                  subclassData);
        
        ObjectOutputStream out = new
            ObjectOutputStream(new FileOutputStream(file));
        
        out.writeObject(data);
        out.flush();
        out.close();
    }
    
    // restore should *only* be called once, at startup.
    protected void restore() throws IOException, ClassNotFoundException {
        ObjectInputStream in = new
            ObjectInputStream(new FileInputStream(file));
        PersistentData data = 
            (PersistentData) in.readObject();
        
        if (data == null) {
            System.err.println("No data in storage file.");
        } else {
            System.out.println("Restoring!");
            serviceID = data.serviceID;
            leaseMgr = new LeaseRenewalManager();
            luMgr = new LookupDiscoveryManager(data.groups,
                                               data.locs,
                                               null);
            joinMgr = new JoinManager(proxy, data.attrs,
                                      data.serviceID, luMgr,
                                      leaseMgr);
            // Subclasses override restored() to
	    // do subclass-specific restore
	    // behavior.
            this.restored(data.subclassData);
        }
    }

    // subclasses override this to do their own initialization behavior.
    protected void initialize() throws IOException, 
        ClassNotFoundException {
        proxy = getProxy();
        
        if (file.exists()) {
            restore();
        }
        
        if (joinMgr == null) {
            leaseMgr = new LeaseRenewalManager();
            luMgr = new LookupDiscoveryManager(new String[] {""},
                                               null, null);
            joinMgr = new JoinManager(proxy, getAttributes(), 
                                      this, luMgr, leaseMgr);
        }
    }
    
    // subclasses write a main that creates a subclass of B-U-S and
    // calls initialize() on it, then waits.
}
