// This is an application that leases storage "slots" to consumers.

package corejini.chapter13;

import com.sun.jini.lease.landlord.Landlord;
import com.sun.jini.lease.landlord.LandlordLease;
import com.sun.jini.lease.landlord.LeasedResource;
import net.jini.core.lease.Lease;
import net.jini.core.lease.LeaseMapException;
import net.jini.core.lease.UnknownLeaseException;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.Naming;
import java.rmi.server.UnicastRemoteObject;
import java.net.InetAddress;
import java.util.Map;
import java.util.HashMap;
import java.io.Serializable;

class CookieType implements Serializable {
    int index;
    int count;
    
    CookieType(int index, int count) {
        this.index = index;
        this.count = count;
    }
    
    public boolean equals(Object o) {
        if (!(o instanceof CookieType))
            return false;
        
        CookieType other = (CookieType) o;
        
        return other.index == index && other.count == count;
    }
    
    public String toString() {
        return "idx=" + index + ", count=" + count; 
    } 
}

public class SlotProviderImpl extends UnicastRemoteObject 
    implements SlotProvider {
    protected LeasedSlot[] slotMap = new LeasedSlot[8];
    protected MyLandlord landlord = null;
    protected LandlordLease.Factory factory = new LandlordLease.Factory();
    protected static int nextCount = 0;
    
    static class LeasedSlot implements LeasedResource {
        protected CookieType cookie;
        protected long expiration;
        protected Object value;
        
        public LeasedSlot(int index) {
            this(index, -1);
        }
        public LeasedSlot(int index, int count) {
            this.cookie = new CookieType(index, count);
	    this.expiration = 0;
        }
        public Object getCookie() {
            return cookie;
        }
        public long getExpiration() {
            return expiration;
        }
        public void setExpiration(long expiration) {
            this.expiration = expiration;
        }
        Object getValue() {
            return value;
        }
        void setValue(Object value) {
            this.value = value;
        }
        void setCookieCount(int count) {
            cookie.count = count;
        }
        int getCookieCount() {
            return cookie.count;
        }
    }
                                                
    
    class MyLandlord extends UnicastRemoteObject implements Landlord {
        protected static final int MAX_LEASE = 1000 * 60 * 10;
        
        public MyLandlord() throws RemoteException {
            super();
        }
        
        public long getRealDuration(long duration) {
            if (duration > MAX_LEASE || duration == Lease.ANY) {
                return MAX_LEASE;
            } else {
                return duration;
            }
        }

        public void cancel(Object cookie) throws UnknownLeaseException {
            long currentTime = System.currentTimeMillis();
            
            System.out.println("+++ landlord cancel: " + cookie);

	    LeasedSlot slot = validateSlot(cookie);
            
            slot.setCookieCount(-1);
            slot.setValue(null);
        }

        public Map cancelAll(Object[] cookies) {
            System.out.println("landlord cancel all: ");
            Map exceptionMap = null;
         //   LeaseMapException lme = null;

            for (int i=0 ; i<cookies.length ; i++) {
                try {
                    cancel(cookies[i]);
                } catch (UnknownLeaseException ex) {
                    if (exceptionMap == null) {
                        exceptionMap = new HashMap();
                        exceptionMap.put(cookies[i], ex);
                    }
                }
            }
            
            return exceptionMap;
        }

        public long renew(Object cookie, long extension) 
	    throws UnknownLeaseException {
            System.out.println("+++ landlord renew " + cookie + 
			       " for " + extension);
            
	    LeasedSlot slot = validateSlot(cookie);
            
            extension = getRealDuration(extension);
            slot.setExpiration(extension + System.currentTimeMillis());
            return extension;
        }
        public Landlord.RenewResults renewAll(Object[] cookies,
                                              long[] extensions) {
            long[] granted = new long[cookies.length];
            Exception[] denied = null;
                
            for (int i=0 ; i<cookies.length ; i++) {
                try {
                    long result = renew(cookies[i], extensions[i]);
                    granted[i] = result;
                } catch (Exception ex) {
                    if (denied == null) {
                        denied = new Exception[cookies.length + 1];
                    }
                    denied[i+1] = ex;
		    granted[i] = -1;
                }
            }
            
            Landlord.RenewResults results = 
                new Landlord.RenewResults(granted, denied);
            return results;
        }
    }

    public SlotProviderImpl() throws RemoteException {
        super();
        
        for (int i=0 ; i<slotMap.length ; i++) {
            slotMap[i] = new LeasedSlot(i);
        }
        
        landlord = new MyLandlord();
    }
    
    public synchronized SlotResult leaseFreeSlot(Object value, long duration)
        throws RemoteException {
        long currentTime = System.currentTimeMillis();
        duration = landlord.getRealDuration(duration);
        
        // return the first unused free slot
        for (int i=0 ; i<slotMap.length ; i++) {
	    LeasedSlot slot = slotMap[i];

            if (slot.getCookieCount() == -1 || 
		slot.getExpiration() < currentTime) {
                slot.setExpiration(duration + currentTime);
                slot.setValue(value);
                slot.setCookieCount(nextCount++);

                System.out.println("+++ setting slot " + i + " to " + value);
                System.out.println("    cookie is " + slot.getCookie());

                Lease lease = factory.newLease(slot.getCookie(), landlord,
                                               duration + currentTime);
                return new SlotResult(i, lease, slot.getCookie());
            }
        }
        
        // no available slots
        return null;
    }

    public synchronized Object getValue(Object key) throws RemoteException,
    UnknownLeaseException {
	LeasedSlot slot = validateSlot(key);

	return slot.getValue();
    }

    LeasedSlot validateSlot(Object cookie) throws UnknownLeaseException {
	long currentTime = System.currentTimeMillis();

	if (!(cookie instanceof CookieType)) {
	    throw new UnknownLeaseException("Bad cookie type"); 
	} 
            
	int index = ((CookieType) cookie).index;
	int count = ((CookieType) cookie).count;
            
	if (index < 0 || index >= slotMap.length) {
	    throw new UnknownLeaseException("Bad cookie data");
	}

	LeasedSlot slot = slotMap[index];
            
	if (slot.getCookieCount() == -1) {
	    throw new UnknownLeaseException("Slot not under lease");
	}
            
	if (slot.getExpiration() < currentTime) {
	    slot.setCookieCount(-1);
	    slot.setValue(null);
	    throw new UnknownLeaseException("Already expired");
	}
            
	if (!slot.getCookie().equals(cookie)) {
	    throw new UnknownLeaseException("Wrong cookie for slot; " +
					    "slot has been released.");
	}

	return slot;
    }
    
    public static void main(String[] args) {
        if (args.length != 1) {
            System.err.println("Usage:  SlotProvider <binding_string>");
            System.exit(1);
        }

	if (System.getSecurityManager() == null) {
	    System.setSecurityManager(
		    new RMISecurityManager());
	}
        
        try {
            String host = InetAddress.getLocalHost().getHostName();
            String url = "rmi://" + host + "/" + args[0];
            SlotProviderImpl spi = new SlotProviderImpl();
            Naming.rebind(url, spi);
            System.out.println("SlotProvider bound to " + url);
        } catch (Exception ex) {
            System.err.println("Couldn't create SlotProvider: " +
                               ex.getMessage());
            System.exit(1);
        }
    }
}
