// A service back-end to manage leases.

package corejini.chapter12;

import net.jini.core.lease.Lease;
import net.jini.core.lease.UnknownLeaseException;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.UnknownEventException;
import net.jini.core.discovery.LookupLocator;
import net.jini.core.lookup.ServiceID;
import net.jini.core.entry.Entry;
import net.jini.lookup.JoinManager;
import net.jini.lookup.ServiceIDListener;
import net.jini.discovery.LookupDiscoveryManager;
import net.jini.lease.LeaseRenewalManager;
import net.jini.lease.LeaseRenewalEvent;
import net.jini.lease.LeaseListener;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import java.util.HashMap;
import java.util.Iterator;
import java.io.File;
import java.io.Serializable;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.FileOutputStream;
import java.io.FileInputStream;

public class LeaseService  extends UnicastRemoteObject 
    implements RemoteLeaseRenewer, ServiceIDListener {
    protected JoinManager joinMgr = null;
    protected LeaseRenewalManager leaseMgr = 
        new LeaseRenewalManager();
    protected String storageLoc = null;
    protected LeaseRenewalService proxy;
    protected HashMap allLeases = new HashMap();
    protected File file = null;
    protected ServiceID serviceID = null;
    protected long eventID = 0;

    // Maps from local lease listeners to remote event 
    // listeners
    class LeaseListenerConduit implements LeaseListener {
        RemoteEventListener remote;
        long eventID;
        long seqNum = 0;
    
        LeaseListenerConduit(RemoteEventListener remote, 
                             long eventID) {
            this.remote = remote;
            this.eventID = eventID;
        }
        
        public void notify(LeaseRenewalEvent ev) {
            // Always remove the lease from the set of leases
            allLeases.remove(ev.getLease());
            try {
                checkpoint();
            } catch (IOException ex) {
                System.err.println("Couldn't checkpoint to " + storageLoc +
                                   ": " + ex.getMessage());
                System.err.println("Service not removed from storage.");
            }
            
            try {
                if (remote != null) {
                    remote.notify(remotify(ev, eventID, seqNum++));
                }
            } catch (RemoteException ex) {
                System.err.println("Couldn't notify remote listener: " +
                                   ex.getMessage());
            } catch (UnknownEventException ex) {
                System.err.println("UnknownEvent: " +
                                   ex.getMessage());
                remote = null;
            }
        }
        
        // Produce a RemoteLeaseRenewalEvent from a
        // local LeaseRenewalEvent.
        RemoteLeaseRenewalEvent remotify(LeaseRenewalEvent ev,
                                         long eventID, long seqNum) {
            return new RemoteLeaseRenewalEvent(eventID,
                                               seqNum,
                                               proxy,
                                               ev.getLease(),
                                               ev.getExpiration(),
                                               ev.getException());
        }
    }
    
    // This is the state that we need to keep around to
    // manage leases.
    static class LeaseEntry implements Serializable {
        long expiration;
        RemoteEventListener listener;
        long id;    // ID to use when sending events
        
        LeaseEntry(long expiration, RemoteEventListener listener,
                   long id) {
            this.expiration = expiration;
            this.listener = listener;
            this.id = id;
        }
    }
    
    // All of this service's persistent data...
    static class LeaseServiceData implements Serializable {
        ServiceID id;
        Entry[] attrs;
        String[] groups;
        LookupLocator[] locs;
        HashMap leases;
        
        LeaseServiceData(ServiceID id, Entry[] attrs, String[] groups,
                         LookupLocator[] locs, HashMap leases) {
            this.id = id;
            this.attrs = attrs;
            this.groups = groups;
            this.locs = locs;
            this.leases = leases;
        }
    }
    
    public LeaseService(String storageLoc) 
        throws RemoteException, IOException, 
        ClassNotFoundException {
        if (System.getSecurityManager() == null) {
            System.setSecurityManager(
                      new RMISecurityManager());
        }
        
        this.storageLoc = storageLoc;
        
        proxy = new LeaseServiceProxy(this);
        
        file = new File(storageLoc);
        
        // If the file exists, treat this as a re-start.  Otherwise
        // it's the first time the service has been run.
        if (file.exists()) {
            restore();
        }
        
        // We'll use the "initial" form of the join manager ctor
        // if the file didn't exist or restore() didn't work.
        if (joinMgr == null) {
            LookupDiscoveryManager luMgr =
		new LookupDiscoveryManager(new String[]{ "" },
					   null,
					   null);
            joinMgr = new JoinManager(proxy, null, this, 
                                      luMgr, null);
        }
    }
    
    protected void checkpoint() throws IOException {
        LeaseServiceData data;
        
        // Cycle through allLeases and chuck the ones that have
        // expired.
        Iterator keys = allLeases.keySet().iterator();
        while (keys.hasNext()) {
            Lease lease = (Lease) keys.next();
            LeaseEntry entry = (LeaseEntry) allLeases.get(lease);
            if (entry.expiration >= System.currentTimeMillis())
                allLeases.remove(lease);
        }
        
        LookupDiscoveryManager luMgr =
            (LookupDiscoveryManager) joinMgr.getDiscoveryManager();
        data = new LeaseServiceData(serviceID,
                                    joinMgr.getAttributes(),
                                    luMgr.getGroups(),
                                    luMgr.getLocators(),
                                    allLeases);
        ObjectOutputStream out = new
            ObjectOutputStream(new FileOutputStream(file));
        
        out.writeObject(data);
        out.flush();
        out.close();
    }
    
    protected void restore() throws IOException, ClassNotFoundException {
        ObjectInputStream in = new
            ObjectInputStream(new FileInputStream(file));
        LeaseServiceData data = (LeaseServiceData) in.readObject();
        
        if (data == null) {
            System.out.println("No data in data file, although it exists.");
        } else {
            allLeases = data.leases;
            // fill the lease manager with the leases...
            Iterator iter = allLeases.keySet().iterator();
            while (iter.hasNext()) {
                Lease lease = (Lease) iter.next();
                LeaseEntry entry = (LeaseEntry) allLeases.get(lease);
                
                if (entry.expiration >= System.currentTimeMillis()) {
                    allLeases.remove(lease);
                    continue;
                }
                
                leaseMgr.renewUntil(lease, entry.expiration,
                                new LeaseListenerConduit(entry.listener,
                                                         entry.id));

            }
            serviceID = data.id;
            LookupDiscoveryManager luMgr =
                new LookupDiscoveryManager(data.groups,
                                           data.locs,
                                           null);
            joinMgr = new JoinManager(proxy, data.attrs,
                                      data.id, luMgr, null);
        }
    }
    
    public void serviceIDNotify(ServiceID serviceID) {
        this.serviceID = serviceID;
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("Service ID not saved.");
        }
    }
    
    // these are the methods of RemoteLeaseListener that our proxy
    // will use to communicate with us.
    
    public void renewAny(Lease lease, RemoteEventListener listener)
        throws RemoteException {
        long id = eventID++;
        lease.setSerialFormat(Lease.ABSOLUTE);
        allLeases.put(lease, new LeaseEntry(Lease.ANY, listener, id));
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("New lease not saved.");
        }
        
        LeaseListenerConduit conduit =
            new LeaseListenerConduit(listener, id);
        
        leaseMgr.renewUntil(lease, Lease.ANY, conduit);
    }

    public void renewFor(Lease lease, long duration,
                         RemoteEventListener listener)
        throws RemoteException {
        long id = eventID++;
        lease.setSerialFormat(Lease.ABSOLUTE);
        allLeases.put(lease, 
                      new LeaseEntry(duration + System.currentTimeMillis(), 
                                     listener, id));
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("New lease not saved.");
        }
        
        LeaseListenerConduit conduit =
            new LeaseListenerConduit(listener, id);
        
        leaseMgr.renewFor(lease, duration, conduit);
    }
    
    public void clear() throws RemoteException {
        allLeases.clear();
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("Lease storage not cleared.");
        }
        
        leaseMgr.clear();
    }
    public void remove(Lease lease) throws RemoteException,
        UnknownLeaseException {
        allLeases.remove(lease);
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("Lease not removed from storage.");
        }
        
        leaseMgr.remove(lease);
    }
    public void cancel(Lease lease) throws RemoteException,
            UnknownLeaseException {
        allLeases.remove(lease);
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("Lease not removed from storage.");
        }
        
        leaseMgr.cancel(lease);
    }

    // Note that absolute/relative time conversions.
    public long getRemainingDuration(Lease lease) throws RemoteException,
                    UnknownLeaseException {
        long expiration = leaseMgr.getExpiration(lease);
        return expiration - System.currentTimeMillis();
    }

    public void setRemainingDuration(Lease lease, long duration)
        throws RemoteException, UnknownLeaseException {
        LeaseEntry entry = (LeaseEntry) allLeases.get(lease);
        long expiration = System.currentTimeMillis() + duration;
        
        if (entry != null) {
            entry.expiration = expiration;
            allLeases.put(lease, entry);
        }
        
        try {
            checkpoint();
        } catch (IOException ex) {
            System.err.println("Couldn't checkpoint to " + storageLoc +
                               ": " + ex.getMessage());
            System.err.println("Lease expiration not updated in storage.");
        }

        leaseMgr.setExpiration(lease, expiration);
    }

    // For now, don't return an administration object.
    public Object getAdmin() {
        return null;
    }
    
    public static void main(String[] args) {
        if (args.length != 1) {
            System.err.println("Usage: LeaseService <storage_loc>");
            System.exit(1);
        }
        
        try {
            LeaseService src = new LeaseService(args[0]);
            while (true) {
                try {
                    Thread.sleep(Long.MAX_VALUE);
                } catch (InterruptedException ex) {
                }
            }
        } catch (Exception ex) {
            System.out.println("trouble: " + ex.getMessage());
            ex.printStackTrace();
        }
    }
} 
  
