// behavior
//
// lookup single click -- update services
// service single click -- update attrs
// attr single click -- noop
//
// lookup double click -- bring up admin panel
// service double click -- bring up admin panel
// attr double click -- noop

package corejini.chapter11;

import net.jini.admin.Administrable;
import net.jini.discovery.DiscoveryGroupManagement;
import net.jini.discovery.LookupDiscoveryManager;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceItem;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.lookup.ServiceMatches;
import net.jini.core.event.EventRegistration;
import net.jini.lookup.entry.EntryBean;
import net.jini.lookup.entry.EntryBeans;
import net.jini.core.entry.Entry;
import net.jini.lease.LeaseRenewalManager;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JDialog;
import java.awt.event.MouseListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowAdapter;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.util.ArrayList;
import java.io.IOException;
import java.rmi.RemoteException;
import java.rmi.RMISecurityManager;
    
// This is the overall browser application.  It's a panel,
// so it can be embedded in other JComponents
public class Browser extends JPanel {
    // Models for the three panes
    protected ArrayListModel lookups, services, attrs;
    protected ServiceRegistrar currentLookup = null;
    // Search for everything
    protected ServiceTemplate template = 
        new ServiceTemplate(null, null, null);
    // Current event registration
    protected EventRegistration er = null;
    protected Listener eventListener;
    protected LeaseRenewalManager leaseMgr = new LeaseRenewalManager();
    protected LookupDiscoveryManager disco;
    
    public Browser() throws IOException, RemoteException { 
        MouseListener listener = new ListClickListener(this);
        
        // Make an event listener
        eventListener = new Listener(this);
        
        // Create the list models
        lookups = new ArrayListModel();
        services = new ArrayListModel();
        attrs = new ArrayListModel();
        
        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        
        setLayout(layout);
        
        c.fill = GridBagConstraints.BOTH;
        c.weightx = 1.0;
        c.gridx = GridBagConstraints.RELATIVE;
        c.gridy = 0;
        
        // do labels across the top
        JLabel label = new JLabel("Lookup Services");
        layout.setConstraints(label, c);
        add(label);
        
        label = new JLabel("Services");
        layout.setConstraints(label, c);
        add(label);
        
        label = new JLabel("Attributes");
        layout.setConstraints(label, c);
        add(label);
        
        // do scrolling lists across the bottom
        c.gridy = 1;
        c.weighty = 1.0;
        JList list = new JList(lookups);
        list.addMouseListener(listener);
        list.setCellRenderer(new LookupCellRenderer());
        JScrollPane scroller = new JScrollPane(list);
        layout.setConstraints(scroller, c);
        add(scroller);
        
        list = new JList(services);
        list.setCellRenderer(new ServiceCellRenderer());
        list.addMouseListener(listener);
        scroller = new JScrollPane(list);
        layout.setConstraints(scroller, c);
        add(scroller);
        
        list = new JList(attrs);
        list.setCellRenderer(new AttrCellRenderer());
        list.addMouseListener(listener);
        scroller = new JScrollPane(list);
        layout.setConstraints(scroller, c);
        add(scroller);

        // Set up for discovery
        disco = new LookupDiscoveryManager(
                DiscoveryGroupManagement.ALL_GROUPS,
                null,
		new Discoverer(this));
    }
    
    ArrayListModel getLookups() {
        return lookups;
    }
    ArrayListModel getServices() {
        return services;
    }
    ArrayListModel getAttrs() {
        return attrs;
    }
    
    // Find all services in "reg"
    void updateServices(ServiceRegistrar reg) {
        ServiceMatches matches;
        
        try {
            matches = currentLookup.lookup(template, 
                                           Integer.MAX_VALUE);
        } catch (RemoteException ex) {
            showDialog("Problem contacting registrar:\n" + ex.getMessage());
	    disco.discard(reg);
            return;
        } 
        
        ArrayList newServices = 
            new ArrayList(matches.totalMatches);
        for (int i=0 ; i<matches.totalMatches ; i++) {
            newServices.add(matches.items[i]);
        }
        
        services.setData(newServices);
    }

    // handle single clicks on lookup services--update the
    // list of services
    protected void lookupClicked(int index) {
        unsolicitEvents();
        currentLookup = (ServiceRegistrar) lookups.getElementAt(index);
        updateServices(currentLookup);
        solicitEvents();
    }
    
    // handle double clicks on lookup services--administer
    // the lookup service
    protected void lookupDoubleClicked(int index) {
        currentLookup = (ServiceRegistrar) lookups.getElementAt(index);
        
        if (!(currentLookup instanceof Administrable)) {
            System.out.println("Lookup service is not administrable");
            return;
        }
        
        Object o;
        
        try {
            o = ((Administrable) currentLookup).getAdmin();
        } catch (RemoteException ex) {
            showDialog("Problem contacting registrar:\n" + ex.getMessage());
	    disco.discard(currentLookup);
            return;
        }
        
        if (o != null) {
            System.err.println("GOT AN ADMINABLE REGISTRAR!");
            AdminPanel panel = new AdminPanel(o);
            JFrame frame = new JFrame("Administration");
            frame.getContentPane().add(panel);
            frame.pack();
            frame.setVisible(true);
        }
    }
    
    // handle single clicks on services--update the attr list
    protected void serviceClicked(int index) {
        ServiceItem item = (ServiceItem) services.getElementAt(index);
        ArrayList newEntries = 
            new ArrayList(item.attributeSets.length);
        for (int i=0 ; i<item.attributeSets.length ; i++) {
            newEntries.add(item.attributeSets[i]);
        }
        
        attrs.setData(newEntries);
    }

    // handle double clicks on services--try to administer
    // them
    protected void serviceDoubleClicked(int index) {
        ServiceItem item = (ServiceItem) services.getElementAt(index);
        
        if (!(item.service instanceof Administrable)) {
            System.out.println("Service is not administrable");
            return;
        }
        
        Object o;
        
        try {
            o = ((Administrable) item.service).getAdmin();
        } catch (RemoteException ex) {
            showDialog("Problem contacting registrar:\n" + ex.getMessage());
            return;
        }
        
        if (o != null) {
            AdminPanel panel = new AdminPanel(o);
            JFrame frame = new JFrame("Administration");
            frame.getContentPane().add(panel);
            frame.pack();
            frame.setVisible(true);
        }
        
        System.out.println("Got admin object: " + o);
    }
    
    // Ask a lookup service for events
    void solicitEvents() {
        System.out.println("Soliciting events");
        try {
            er = currentLookup.notify(template, 
                                  ServiceRegistrar.TRANSITION_MATCH_NOMATCH |
                                  ServiceRegistrar.TRANSITION_NOMATCH_MATCH |
                                  ServiceRegistrar.TRANSITION_MATCH_MATCH,
                                  eventListener, null, 10 * 60 * 1000);
        
            leaseMgr.renewUntil(er.getLease(), Long.MAX_VALUE, null);
        } catch (RemoteException ex) {
            er = null;
	    disco.discard(currentLookup);
        }
    }
    
    // Stop receiving events
    void unsolicitEvents() {
        if (er == null)
            return;
        
        System.out.println("UN-Soliciting events");
        try {
            leaseMgr.cancel(er.getLease());
        } catch (Exception ex) {
        }
    }
    
    // Clean up a classname (strip off package part)
    static String declassify(String name) {
        if (name == null) return null;
        
        int idx = name.lastIndexOf('.');
        
        if (idx == -1)
            return name;
        else
            return name.substring(idx+1);
    }
    
    void showDialog(String message) {
        JOptionPane pane = new JOptionPane(message, JOptionPane.DEFAULT_OPTION,
                                           JOptionPane.ERROR_MESSAGE,
                                           null, null, null);
        JDialog dialog = new JDialog(JOptionPane.getFrameForComponent(this), 
                                     "Alert!");
        dialog.getContentPane().setLayout(new BorderLayout());
        dialog.getContentPane().add(pane, BorderLayout.CENTER);
        dialog.pack();
        dialog.show();
    }
    
    static Component getComponentForEntry(Entry e, 
                                          Component defaultComponent) {
        EntryBean bean = null;

	// If we're passed a null entry, return the
	// default component (if it's a label, make 
	// it say "NULL")
	if (e == null) {
	    if (defaultComponent instanceof JLabel) {
	        ((JLabel) defaultComponent).setText("NULL");
            }
	    return defaultComponent;
        }
        
        try {
            bean = EntryBeans.createBean(e);
        } catch (Exception ex) {
        }
                    
        // Now beanInstance will be either null or 
        // will ref a bean. If it's null or it's not 
        // a component, we just tostring it.  If
        // it's a component, we return that.
        if (bean == null || !(bean instanceof Component)) {
            if (defaultComponent instanceof JLabel) {
                String cname = 
                    Browser.declassify(e.getClass().getName());
                ((JLabel) defaultComponent).setText(cname + " [" + 
                                                    e.toString() + "]");
            }
            return defaultComponent;
        } else {
            ((Component) bean).setVisible(true);
            return (Component) bean;
        }
    }
    
    public static void main(String[] args) {
        try {
	    if (System.getSecurityManager() == null) {
		System.setSecurityManager(
                        new RMISecurityManager());
	    }
		
            JFrame frame = new JFrame("Browser");
            frame.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent ev) {
                    System.exit(1);
                }
            });
            frame.getContentPane().add(new Browser());
            frame.pack();
            frame.setVisible(true);
        } catch (Exception ex) {
            System.err.println("Couldn't create browser:" + 
                               ex.getMessage());
            ex.printStackTrace();
        }
    }
}
