// Create an extension of the service that uses
// RMI to communicate with an activatable back-end
// process.

package corejini.appendixc;

import java.io.IOException;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.rmi.Remote;
import java.rmi.MarshalledObject;
import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.activation.Activatable;
import java.rmi.activation.ActivationID;
import java.rmi.activation.ActivationDesc;
import java.rmi.activation.ActivationGroup;
import java.rmi.activation.ActivationGroupID;
import java.rmi.activation.ActivationGroupDesc;
import java.rmi.activation.ActivationException;
import java.util.Properties;
import corejini.chapter5.BackendProtocol;
import corejini.chapter5.HelloWorldServiceInterface;
import corejini.chapter5.HelloWorldServiceWithLeases;
import corejini.chapter5.HelloWorldServiceProxy2;

public class HelloWorldServiceActivatable
    extends HelloWorldServiceWithLeases {
    // Need to pass a policy file to the activation daemon...
    private String policyFile;
    // ...as well as a codebase.
    private String codebase;
    
    // The main difference between this example and
    // HelloWorldServiceRemote (in chapter5) is that
    // in this case the backend RMI object is activatable.
    // So we need a new backend object that extends
    // Activatable but still implements BackendProtocol.
    public static class ActivatableBackend 
        extends Activatable 
        implements BackendProtocol {
        int nextMessage = 0;
        String[] messages = { "Hello, World", 
                              "Goodbye, Cruel World",
                              "What's Up, Doc?" };
        
        public ActivatableBackend(ActivationID id, 
                                  MarshalledObject data) 
            throws RemoteException {
            super(id, 0);
        }
        
        public synchronized String fetchString() 
            throws RemoteException {
            String str =  messages[nextMessage];
            nextMessage = (nextMessage + 1) % messages.length;
            return str;
        }
        
        // should put itself back to sleep after a while.
    } 
    
    // Constructor for the wrapper doesn't have to do
    // anything except let the superclass constructor run
    public HelloWorldServiceActivatable() throws IOException {
    }
    
    public void setPolicyFile(String policyFile) {
        this.policyFile = policyFile;
    }
    public void setCodebase(String codebase) {
        this.codebase = codebase;
    }
    
    protected HelloWorldServiceInterface createProxy() {
        try {
            // Create a descriptor for a new activation 
            // group to run our backend object in.
            Properties props = new Properties();
            props.put("java.security.policy", policyFile);
            ActivationGroupDesc group = 
                new ActivationGroupDesc(props, null);
            // Register the group and get the ID.
            ActivationGroupID gid = 
                ActivationGroup.getSystem().registerGroup(group);
            // Now create the group
            ActivationGroup.createGroup(gid, group, 0);
            
            // Create an activation descriptor
            // for the object
            String location = codebase;
            MarshalledObject data = null;
            ActivationDesc desc = 
                new ActivationDesc("corejini.appendixc." +
                       "HelloWorldServiceBackend.Backend",
                       location, data);
            
            // Create the 'backend' object that will
            // implement the protocol.
            BackendProtocol backend = 
                (BackendProtocol) Activatable.register(desc);
            return new HelloWorldServiceProxy2(backend);
        } catch (RemoteException ex) {
            System.err.println("Error creating backend object: " +
                               ex.getMessage());
            System.exit(1);
            return null;
        } catch (ActivationException ex) {
            System.err.println("Problem with activation: " + 
                               ex.getMessage());
            ex.printStackTrace();
            System.exit(1);
            return null;
        }
    }

    // Create the service and start its lease
    // thread.
    public static void main(String args[]) {
        if (args.length != 2) {
            System.err.println("Usage: HelloWorldServiceActivatable " +
                               " <policy file> <codebase>");
        }
        try {
            HelloWorldServiceActivatable hws = 
                new HelloWorldServiceActivatable();
            // Pass in the policy file and codebase from
            // the command line.
            hws.setPolicyFile(args[0]);
            hws.setCodebase(args[1]);
            // start the lease renewal thread.
            hws.leaseThread = new Thread(hws);
            hws.leaseThread.start();
        } catch (IOException ex) {
            System.out.println("Couldn't create service: " +
                               ex.getMessage());
        }
    }
}
